package com.thalesgroup.marte.rapidrma.export.actions;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.util.Iterator;

import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.rapidrma.export.Activator;
import com.thalesgroup.marte.rapidrma.resource.CPU;
import com.thalesgroup.marte.rapidrma.resource.Node;
import com.thalesgroup.marte.rapidrma.resource.PriorityMappingKind;
import com.thalesgroup.marte.rapidrma.resource.Range;
import com.thalesgroup.marte.rapidrma.resource.Resource;
import com.thalesgroup.marte.rapidrma.resource.ResourceGraph;
import com.thalesgroup.marte.rapidrma.resource.ResourceKind;
import com.thalesgroup.marte.rapidrma.resource.SharedResource;

/**
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 *
 * $Id: ResourceHelper.java,v 1.7 2008/06/16 13:56:12 sdemathieu Exp $
 */

/**
 * This class provides utilties to serialize RapidRMA resource information into
 * related files.
 * 
 * @author Sbastien Demathieu (Thales Research & Technology)
 * @version 1.0 - 26/07/07
 */
public class ResourceHelper extends RapidRMAHelper {

	/**
	 * The resource graph to serialize
	 */
	private ResourceGraph _graph;

	/**
	 * ResourceHelper default constructor.
	 * 
	 * @param name
	 *            the name of the database
	 * @param graph
	 *            the resource graph to serialize
	 */
	public ResourceHelper(String name, ResourceGraph graph) {
		super(name);
		_graph = graph;
		validHashtable();
	}

	/**
	 * Serializes the priority mapping information into a "xxx-pm.rgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	public void createResourcePriorityMapping() throws Exception {

		File file = new File(RG_PM_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);

		if (_graph!=null) {
			Iterator it = _graph.getNode().iterator();
			// Iterates thru nodes
			while (it.hasNext()) {
				Node node = (Node) it.next();
				Iterator itRes = node.getResource().iterator();
				CPU cpu = null;
				boolean found = false;
				// Looks for the CPU associated to the node
				while (itRes.hasNext() && !found) {
					Resource res = (Resource) itRes.next();
					if (res instanceof CPU) {
						cpu = (CPU) res;
						found = true;
					}
				}
				// Writes the mapping information, if there is one.
				Iterator itRange = cpu.getPriorityMapping().iterator();
				int i = 0;
				while (itRange.hasNext()) {
					Range r = (Range) it.next();
				
					RG_PM_Format rg_pm_Format = new RG_PM_Format();
					String[] values = new String[rg_pm_Format.length()];
				
					String id = (String) RapidRMAHelper._idTable.get(node);
					values[rg_pm_Format.getNum(RapidRMAKeywords.INDEX_)] = ""+i;
					values[rg_pm_Format.getNum(RapidRMAKeywords.ID____)] = "R" + id + "_" + i;
					values[rg_pm_Format.getNum(RapidRMAKeywords.RES_ID)] = "R" + id ;
					values[rg_pm_Format.getNum(RapidRMAKeywords.STARTT)] = ""+r.getStart();
					values[rg_pm_Format.getNum(RapidRMAKeywords.END_T_)] = ""+r.getEnd();
				
					rg_pm_Format.write(values,writer);
				}
			}
		}

		writer.close();

	}

	/**
	 * Serializes the resource accesses into a "xxx-acc.rgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	public void createResourceAccess() throws Exception {

		File file = new File(RG_ACC_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);

		if (_graph != null) {
			// Iterates thru nodes
			Iterator itNode = _graph.getNode().iterator();
		
			while (itNode.hasNext()) {
				Node node = (Node) itNode.next();

				RG_ACC_Format rg_acc_Format = new RG_ACC_Format();
				String[] values = new String[rg_acc_Format.length()];
			
				Iterator itAcc = node.getAccesses().iterator();
				while (itAcc.hasNext()) {
					Resource res = (Resource) itAcc.next();

					values[rg_acc_Format.getNum(RapidRMAKeywords.NODEID)] = (String) RapidRMAHelper._idTable.get(node);
					values[rg_acc_Format.getNum(RapidRMAKeywords.A_NODE)] = (String) RapidRMAHelper._idTable.get(res.getNode());
					values[rg_acc_Format.getNum(RapidRMAKeywords.RES_ID)] = (String) RapidRMAHelper._idTable.get(res);

					rg_acc_Format.write(values,writer);
				}
			}
		}

		writer.close();
	}

	/**
	 * Serializes the information related to conclicting resources into a
	 * "xxx-cr.rgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	public void createConflictingResource() throws Exception {

		File file = new File(RG_CR_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);


		if (_graph != null) {
			// Iterates thru nodes
			Iterator nodeIt = _graph.getNode().iterator();
			while (nodeIt.hasNext()) {
				Node node = (Node) nodeIt.next();
				Iterator resIt = node.getResource().iterator();
				while (resIt.hasNext()) {
					Resource res = (Resource) resIt.next();

					if (res instanceof SharedResource) {

						// For each shared resource, check if there is no
						// conflicting resource
						String id = (String) RapidRMAHelper._idTable.get(res);					
						Iterator confList = ((SharedResource) res).getConflicting().iterator();
					
						while (confList.hasNext()) {

							Resource confRes = (Resource) confList.next();

							RG_CR_Format rg_cr_Format1 = new RG_CR_Format();
							String[] values1 = new String[rg_cr_Format1.length()];
							RG_CR_Format rg_cr_Format2 = new RG_CR_Format();
							String[] values2 = new String[rg_cr_Format2.length()];
						
							String confId = (String) RapidRMAHelper._idTable.get(confRes);
						
							values1[rg_cr_Format1.getNum(RapidRMAKeywords.RES_ID)] = "R" + id;
							values1[rg_cr_Format1.getNum(RapidRMAKeywords.CRESID)] = "R" + confId;
							rg_cr_Format1.write(values1,writer);
						
							values2[rg_cr_Format2.getNum(RapidRMAKeywords.RES_ID)] = "R" + confId;
							values2[rg_cr_Format2.getNum(RapidRMAKeywords.CRESID)] = "R" + id;
							rg_cr_Format2.write(values2,writer);
						}
					}
				}
			}
		}

		writer.close();
	}

	/**
	 * Creates the main resource graph in a file.
	 * 
	 * @throws Exception
	 *             if an error occurs during serialization
	 */
	public void createResourceGraph() throws Exception {

		File file = new File(RGFormat.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);
		int i = 2;
		int nodeId = 0;

		// Writes default resources
		RGFormat rgFormat1 = new RGFormat();
		String[] values1 = new String[rgFormat1.length()];
		values1[rgFormat1.getNum(RapidRMAKeywords.DATAID)] = "RESOURCES";
		values1[rgFormat1.getNum(RapidRMAKeywords.INDEX_)] = "0";
		values1[rgFormat1.getNum(RapidRMAKeywords.RES_ID)] = "R0";
		values1[rgFormat1.getNum(RapidRMAKeywords.R_NAME)] = "Free Node";
		values1[rgFormat1.getNum(RapidRMAKeywords.R_TYPE)] = "Node";
		values1[rgFormat1.getNum(RapidRMAKeywords.F_NODE)] = "true";
		values1[rgFormat1.getNum(RapidRMAKeywords.NODE__)] = "Free Node";
		values1[rgFormat1.getNum(RapidRMAKeywords.NODEID)] = "R0";
		values1[rgFormat1.getNum(RapidRMAKeywords.ACQ_T_)] = "0";
		values1[rgFormat1.getNum(RapidRMAKeywords.DEACQT)] = "0";
		values1[rgFormat1.getNum(RapidRMAKeywords.CSRATE)] = "0";
		values1[rgFormat1.getNum(RapidRMAKeywords.P_RATE)] = "1.000000";
		values1[rgFormat1.getNum(RapidRMAKeywords.PRIO_D)] = "0";
		values1[rgFormat1.getNum(RapidRMAKeywords.COMMEN)] = "";
		rgFormat1.write(values1,writer);

		RGFormat rgFormat2 = new RGFormat();
		String[] values2 = new String[rgFormat2.length()];
		values2[rgFormat2.getNum(RapidRMAKeywords.DATAID)] = "RESOURCES";
		values2[rgFormat2.getNum(RapidRMAKeywords.INDEX_)] = "1";
		values2[rgFormat2.getNum(RapidRMAKeywords.RES_ID)] = "R1";
		values2[rgFormat2.getNum(RapidRMAKeywords.R_NAME)] = "CPU";
		values2[rgFormat2.getNum(RapidRMAKeywords.R_TYPE)] = "CPU";
		values2[rgFormat2.getNum(RapidRMAKeywords.F_NODE)] = "false";
		values2[rgFormat2.getNum(RapidRMAKeywords.NODE__)] = "Free Node";
		values2[rgFormat2.getNum(RapidRMAKeywords.NODEID)] = "R0";
		values2[rgFormat2.getNum(RapidRMAKeywords.ACQ_T_)] = "0";
		values2[rgFormat2.getNum(RapidRMAKeywords.DEACQT)] = "0";
		values2[rgFormat2.getNum(RapidRMAKeywords.CSRATE)] = "0";
		values2[rgFormat2.getNum(RapidRMAKeywords.P_RATE)] = "1.000000";
		values2[rgFormat2.getNum(RapidRMAKeywords.PRIO_D)] = "0";
		values2[rgFormat2.getNum(RapidRMAKeywords.COMMEN)] = "";
		rgFormat2.write(values2,writer);
		
		if (_graph!=null) {
			Iterator nodeList = _graph.getNode().iterator();
			while (nodeList.hasNext()) {

				Node node = (Node) nodeList.next();
				writeNode(writer, node, i);
				nodeId = i;
				RapidRMAHelper._idTable.put(node, "R" + i);
				i++;

				Iterator resList = node.getResource().iterator();
				while (resList.hasNext()) {

					Resource res = (Resource) resList.next();
					writeResource(writer, res, node.getName(), nodeId, i);
					RapidRMAHelper._idTable.put(res, "R" + i);
					i++;

				}
			}
		}
		writer.close();
	}

	/**
	 * Helper method that writes node information on a line.
	 * 
	 * @param writer
	 *            the PrintWriter used to write the line
	 * @param node
	 *            the node to serialize
	 * @param index
	 *            the index bound to the node
	 */
	private void writeNode(PrintWriter writer, Node node, int index) {

		RGFormat rgFormat = new RGFormat();
		String[] values = new String[rgFormat.length()];
		
		// Writes a node on a line.
		values[rgFormat.getNum(RapidRMAKeywords.DATAID)] = "RESOURCES";
		values[rgFormat.getNum(RapidRMAKeywords.INDEX_)] = ""+index;
		values[rgFormat.getNum(RapidRMAKeywords.RES_ID)] = "R" + index;
		values[rgFormat.getNum(RapidRMAKeywords.R_NAME)] = ""+node.getName();
		values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Node";
		values[rgFormat.getNum(RapidRMAKeywords.F_NODE)] = "false";
		values[rgFormat.getNum(RapidRMAKeywords.NODE__)] = ""+node.getName();
		values[rgFormat.getNum(RapidRMAKeywords.NODEID)] = "R" + index;

		// Default values for a node
		values[rgFormat.getNum(RapidRMAKeywords.ACQ_T_)] = "0";
		values[rgFormat.getNum(RapidRMAKeywords.DEACQT)] = "0";
		values[rgFormat.getNum(RapidRMAKeywords.CSRATE)] = "0";
		values[rgFormat.getNum(RapidRMAKeywords.P_RATE)] =  "1.0";

		// EM ???
		// Priority mapping order
		Iterator it = node.getResource().iterator();
		while (it.hasNext()) {
			Object obj = it.next();
			if (obj instanceof CPU) {
				CPU cpu = (CPU) obj;
				if (cpu.getMappingKind().equals(
						PriorityMappingKind.LOW_NUMBER_LOW_PRIORITY_LITERAL)) {
					values[rgFormat.getNum(RapidRMAKeywords.PRIO_D)] =  "0";
				} else {
					values[rgFormat.getNum(RapidRMAKeywords.PRIO_D)] =  "1";
				}
			}
		}

		// Comments
		values[rgFormat.getNum(RapidRMAKeywords.COMMEN)] = "";

		rgFormat.write(values,writer);
	}

	/**
	 * Writes information related to a resource on a line.
	 * 
	 * @param writer
	 *            The printwriter that access to the file
	 * @param res
	 *            The resource to serialize
	 * @param nodeName
	 *            the name of the node that relates to the resource
	 * @param nodeId
	 *            the id of the node that relates to the resource
	 * @param index
	 *            the current resource index (used to generate ids)
	 */
	private void writeResource(PrintWriter writer, Resource res,
			String nodeName, int nodeId, int index) {

		RGFormat rgFormat = new RGFormat();
		String[] values = new String[rgFormat.length()];
		
		// Writes a resource on a line.
		values[rgFormat.getNum(RapidRMAKeywords.DATAID)] = "RESOURCES";
		values[rgFormat.getNum(RapidRMAKeywords.INDEX_)] = ""+index;
		values[rgFormat.getNum(RapidRMAKeywords.RES_ID)] = "R" + index;
		values[rgFormat.getNum(RapidRMAKeywords.R_NAME)] = res.getName();

		if (res instanceof CPU) {
			values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "CPU";
		} else {
			SharedResource shared = (SharedResource) res;
			if (shared.getType().equals(ResourceKind.METHOD_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Method";
			} else if (shared.getType().equals(ResourceKind.DB_LOCK_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Database Lock";
			} else if (shared.getType().equals(ResourceKind.THREAD_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Thread";
			} else if (shared.getType().equals(ResourceKind.DSP_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "DSP";
			} else if (shared.getType().equals(ResourceKind.DSP_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "DSP";
			} else if (shared.getType().equals(ResourceKind.IO_PORT_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "IO Port";
			} else if (shared.getType().equals(ResourceKind.VME_VSB_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "VME/VSB";
			} else if (shared.getType().equals(ResourceKind.SEMAPHORE_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Semaphore";
			} else if (shared.getType().equals(ResourceKind.VME_FDDI_DL_LITERAL)) { //EM 30/07/2007
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "VME/FDDI/DL";
			} else if (shared.getType().equals(ResourceKind.VME_FDDI_DL_LITERAL)) {//EM 30/07/2007
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "VME/FDDI/DL";
			} else if (shared.getType().equals(ResourceKind.MEMORY_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Memory";
			} else if (shared.getType().equals(ResourceKind._422_561_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "422/561";
			} else if (shared.getType().equals(ResourceKind.OTHER_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Other";
			} else if (shared.getType().equals(ResourceKind.SERVICE_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Service";
			} else if (shared.getType().equals(ResourceKind.FDDI_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "FDDI";
			} else if (shared.getType().equals(ResourceKind.DMA_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "DMA";
			} else if (shared.getType().equals(ResourceKind.RECEIVER_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Receiver";
			} else if (shared.getType().equals(ResourceKind.SENSOR_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Sensor";
			} else if (shared.getType().equals(ResourceKind.PANEL_LITERAL)) {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Panel";
			} else {
				values[rgFormat.getNum(RapidRMAKeywords.R_TYPE)] = "Bus";
			}
		}
		values[rgFormat.getNum(RapidRMAKeywords.F_NODE)] = "false";

		values[rgFormat.getNum(RapidRMAKeywords.NODE__)] = nodeName;

		values[rgFormat.getNum(RapidRMAKeywords.NODEID)] = "R" + nodeId;

		if (res instanceof SharedResource) {
			values[rgFormat.getNum(RapidRMAKeywords.ACQ_T_)] = ""+((SharedResource) res).getAcqTime();
		} else {
			values[rgFormat.getNum(RapidRMAKeywords.ACQ_T_)] = "0";
		}

		if (res instanceof SharedResource) {
			values[rgFormat.getNum(RapidRMAKeywords.DEACQT)] = ""+((SharedResource) res).getDeAcqTime();
		} else {
			values[rgFormat.getNum(RapidRMAKeywords.DEACQT)] = "0";
		}

		if ((res instanceof CPU) && (((CPU) res).getContextSwRate() != null)) {
			values[rgFormat.getNum(RapidRMAKeywords.CSRATE)] = ""+((CPU) res).getContextSwRate();
		} else {
			values[rgFormat.getNum(RapidRMAKeywords.CSRATE)] = "0";
		}

		if ((res instanceof CPU) && (((CPU) res).getContextSwRate() != null)) {
			values[rgFormat.getNum(RapidRMAKeywords.P_RATE)] = ""+((CPU) res).getProcessingRate();
		} else {
			values[rgFormat.getNum(RapidRMAKeywords.P_RATE)] = "1.0";
		}

		// Priority mapping - useless for resources
		values[rgFormat.getNum(RapidRMAKeywords.PRIO_D)] = "0";

		// Comments
		values[rgFormat.getNum(RapidRMAKeywords.COMMEN)] = "";

		rgFormat.write(values,writer);
	}
	
	public void create() {
		try {
			createResourceGraph();

			/* Creates the priority mapping file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Priority mapping file");
			createResourcePriorityMapping();

			/* Creates the resource access file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Resource access file");
			createResourceAccess();

			/* Creates the conflicting resource file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Conflicting resource file");
			createConflictingResource();

			// ...
		}
		catch (Exception e) {
			Log.errorMessage(Activator.PLUGIN_ID,"Problem in creating ResourceHelper",e);
		}
	}

}
