/**
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 *
 * $Id: TaskHelper.java,v 1.9 2008/06/16 13:56:12 sdemathieu Exp $
 */

package com.thalesgroup.marte.rapidrma.export.actions;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.rapidrma.export.Activator;
import com.thalesgroup.marte.rapidrma.resource.CPU;
import com.thalesgroup.marte.rapidrma.resource.Node;
import com.thalesgroup.marte.rapidrma.resource.Resource;
import com.thalesgroup.marte.rapidrma.task.DeadlineKind;
import com.thalesgroup.marte.rapidrma.task.Deterministic;
import com.thalesgroup.marte.rapidrma.task.Distribution;
import com.thalesgroup.marte.rapidrma.task.Exponential;
import com.thalesgroup.marte.rapidrma.task.IntermediateDeadline;
import com.thalesgroup.marte.rapidrma.task.NonPremptibleSection;
import com.thalesgroup.marte.rapidrma.task.PeriodicServer;
import com.thalesgroup.marte.rapidrma.task.QueueingKind;
import com.thalesgroup.marte.rapidrma.task.ResourceUsage;
import com.thalesgroup.marte.rapidrma.task.ServerKind;
import com.thalesgroup.marte.rapidrma.task.Task;
import com.thalesgroup.marte.rapidrma.task.TaskFactory;
import com.thalesgroup.marte.rapidrma.task.TaskGraph;
import com.thalesgroup.marte.rapidrma.task.Uniform;

/**
 * This class provides utilties to serialize RapidRMA task information into a
 * file.
 * 
 * @author Sbastien Demathieu (Thales Research & Technology)
 * @version 1.0 - 26/07/07
 * 
 */
public class TaskHelper extends RapidRMAHelper{


	/**
	 * The task graph to serialize
	 */
	private TaskGraph _graph;
	
	/**
	 * Global index
	 */
	private int globalIndex = 0;

	/**
	 * TaskHelper default constructor
	 * 
	 * @param name
	 *            the name of the database
	 * @param graph
	 *            the task graph to serialize
	 */
	public TaskHelper(String name, TaskGraph graph) {
		super(name);
		_graph = graph;
		validHashtable();
	}

	/**
	 * Creates a RapidRMA task graph file.
	 * 
	 * Note: it looks that RapidRMA stores part of its analysis results in
	 * fields that cannot be editable thru the GUI. The later are not supported
	 * so far but it can be an useful some day.
	 * 
	 * @throws Exception
	 *             if an error occurs during serialization
	 */
	private void createTaskGraph() throws Exception {

		File file = new File(TGFormat.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);


		if (_graph != null) {
			Iterator taskIt = _graph.getTask().iterator();
			int index = globalIndex;
			while (taskIt.hasNext()) {

				Task task = (Task) taskIt.next();

				TGFormat tgFormat = new TGFormat();
				String[] values = new String[tgFormat.length()];
			
				// Index
				values[tgFormat.getNum(RapidRMAKeywords.INDEX_)] = ""+index;
			
				// Task identifier
				values[tgFormat.getNum(RapidRMAKeywords.TASKID)] = "T" + index;
				RapidRMAHelper._idTable.put(task, "T" + index);
			
				// Task name
				values[tgFormat.getNum(RapidRMAKeywords.T_NAME)] = task.getName();

				Distribution period = task.getPeriod();
				if (period!=null) {
					// Period
					if (period instanceof Deterministic) {
						values[tgFormat.getNum(RapidRMAKeywords.PERIOD)] = "Det: (" + period.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.P_SELE)] = "Deterministic";
						values[tgFormat.getNum(RapidRMAKeywords.P_MIN_)] = ""+period.getValue();
						values[tgFormat.getNum(RapidRMAKeywords.P_MAX_)] = ""+period.getValue();
						values[tgFormat.getNum(RapidRMAKeywords.P_AVG_)] = ""+period.getValue();
					} else if (period instanceof Uniform) {
						values[tgFormat.getNum(RapidRMAKeywords.PERIOD)] = "Unf: (" + period.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.P_SELE)] = "Uniform";
						values[tgFormat.getNum(RapidRMAKeywords.P_MIN_)] = ""+((Uniform) period).getMin();
						values[tgFormat.getNum(RapidRMAKeywords.P_MAX_)] = ""+((Uniform) period).getMax();
						values[tgFormat.getNum(RapidRMAKeywords.P_AVG_)] = ""+period.getValue();
					} else if (period instanceof Exponential) {
						values[tgFormat.getNum(RapidRMAKeywords.PERIOD)] = "Exp: (" + period.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.P_SELE)] = "Exponential";
						values[tgFormat.getNum(RapidRMAKeywords.P_MIN_)] = ""+((Exponential) period).getMin();
						values[tgFormat.getNum(RapidRMAKeywords.P_MAX_)] = "2147483647"; // Value generated by RapidRMA... Purpose?..
						values[tgFormat.getNum(RapidRMAKeywords.P_AVG_)] = ""+((Exponential) period).getAvg();
					}
					else {
						Log.warningMessage(Activator.PLUGIN_ID,"The period defined for "+task.getName()+" is not correct");
					}
					values[tgFormat.getNum(RapidRMAKeywords.PHNAME)] = "";
					values[tgFormat.getNum(RapidRMAKeywords.PCOFFP)] = "0";
				}
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"No period is defined for the task "+task.getName());
				}

				Distribution work = task.getAmountOfWork();
				// Amount of Work
				if (work!=null) {
					if (work instanceof Deterministic) {
						values[tgFormat.getNum(RapidRMAKeywords.AOWORK)] = "Det: (" + work.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.AOWSEL)] = "Deterministic";
						values[tgFormat.getNum(RapidRMAKeywords.AOWMIN)] = ""+work.getValue();
						values[tgFormat.getNum(RapidRMAKeywords.AOWMAX)] = ""+work.getValue();
						values[tgFormat.getNum(RapidRMAKeywords.AOWAVG)] = ""+work.getValue();
					} else if (work instanceof Uniform) {
						values[tgFormat.getNum(RapidRMAKeywords.AOWORK)] = "Unf: (" + work.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.AOWSEL)] = "Uniform";
						values[tgFormat.getNum(RapidRMAKeywords.AOWMIN)] = ""+((Uniform) work).getMin();
						values[tgFormat.getNum(RapidRMAKeywords.AOWMAX)] = ""+((Uniform) work).getMax();
						values[tgFormat.getNum(RapidRMAKeywords.AOWAVG)] = ""+work.getValue();
					} else if (period instanceof Exponential) {
						values[tgFormat.getNum(RapidRMAKeywords.AOWORK)] = "Exp: (" + work.getValue() + ")";
						values[tgFormat.getNum(RapidRMAKeywords.AOWSEL)] = "Exponential";
						values[tgFormat.getNum(RapidRMAKeywords.AOWMIN)] = ""+((Exponential) work).getMin();
						values[tgFormat.getNum(RapidRMAKeywords.AOWMAX)] = "2147483647"; // Value generated by RapidRMA... Purpose?..
						values[tgFormat.getNum(RapidRMAKeywords.AOWAVG)] = ""+((Exponential) work).getAvg();
					}
					else {
						Log.warningMessage(Activator.PLUGIN_ID,"The work defined for "+task.getName()+" is not correct");
					}
					values[tgFormat.getNum(RapidRMAKeywords.AOW_HN)] = "";
					values[tgFormat.getNum(RapidRMAKeywords.AOWCOP)] = "0";
				}	
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"No period is defined for the task "+task.getName());
				}

				// Ready time
				values[tgFormat.getNum(RapidRMAKeywords.READYT)] = ""+task.getReadyTime();

				// Relative deadline
				values[tgFormat.getNum(RapidRMAKeywords.R_DEAD)] = ""+task.getRelDeadline();

				// Drop deadline
				values[tgFormat.getNum(RapidRMAKeywords.DROPDL)] = ""+task.getDropDeadline();

				// Phase
				values[tgFormat.getNum(RapidRMAKeywords.PHASE_)] = ""+task.getPhase();

				// Priority
				values[tgFormat.getNum(RapidRMAKeywords.PRIO__)] = ""+task.getPriority();

				// Network QoS
				values[tgFormat.getNum(RapidRMAKeywords.NW_QOS)] = ""+task.getNetworkQoS();

				Resource activeResource = task.getActiveResource();
				// Active resource
				if (activeResource !=null) {
					values[tgFormat.getNum(RapidRMAKeywords.A_RES_)] = (String) RapidRMAHelper._idTable.get(activeResource);
				}
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"The task "+task.getName()+" has no active resource");
				}
			
				/*
				 * The fact that a task runs on a CPU is an implicit
				 * kind of resource usage in the RapidRMA tool and meta-model. When
				 * serializing we need to make explicit these usages.
				 */
				File trFile = new File(TG_TR_Format.FILENAME);
				FileOutputStream trOut = new FileOutputStream(trFile,true);
				PrintWriter trWriter = new PrintWriter(trOut,true);
				ResourceUsage cpuUsage = TaskFactory.eINSTANCE.createResourceUsage();
				cpuUsage.setResource(task.getActiveResource());
				writeResourceUsage(cpuUsage, "T" + index, trWriter);
				trWriter.close();

				// Deadline type
				if (task.getDeadline().equals(DeadlineKind.HARD_DEADLINE_LITERAL)) {
					values[tgFormat.getNum(RapidRMAKeywords.DLTYPE)] = ("Hard Deadline");
				} else if (task.getDeadline().equals(
					DeadlineKind.SOFT_DEADLINE_LITERAL)) {
					values[tgFormat.getNum(RapidRMAKeywords.DLTYPE)] = "Soft Deadline";
				} else {
					values[tgFormat.getNum(RapidRMAKeywords.DLTYPE)] = "No Deadline";
				}

				// Instances
				values[tgFormat.getNum(RapidRMAKeywords.INST__)] = ""+task.getInstances();

				PeriodicServer periodicServer = task.getServer();
				// Periodic server
				if (periodicServer != null) {
					values[tgFormat.getNum(RapidRMAKeywords.T_SERV)] = (String) RapidRMAHelper._idTable.get(periodicServer);
				} else {
					values[tgFormat.getNum(RapidRMAKeywords.T_SERV)] = "";
				}

				String comments = task.getComments();
				// Comment
				if (comments != null) {
					values[tgFormat.getNum(RapidRMAKeywords.COMMEN)] = comments;
				} else {
					values[tgFormat.getNum(RapidRMAKeywords.COMMEN)] = "";
				}
			
				// Spare capacity calc required
				values[tgFormat.getNum(RapidRMAKeywords.SCCREQ)] = "false";

				// Coordinates
				values[tgFormat.getNum(RapidRMAKeywords.XCOORD)] = ""+task.getXCoord();
				values[tgFormat.getNum(RapidRMAKeywords.YCOORD)] = ""+task.getYCoord();

				tgFormat.write(values,writer);
				index++;
			}
			globalIndex = index;
		}

		writer.close();
	}

	/**
	 * Helper method that writes an explicit resource usage (for a passive
	 * resource) or an implicit one (for a task or a server) on a print writer.
	 * 
	 * @param usage
	 *            the ResourceUsage to writer. The later may be owned by the
	 *            model or it be created by the serializer to indicate the
	 *            implicit use of CPUs by tasks and servers.
	 * @param accessId
	 *            the identifier of the related task or server
	 * @param writer
	 *            the print writer
	 */
	private void writeResourceUsage(ResourceUsage usage, String accessId,
			PrintWriter writer) {
		
		TG_TR_Format tg_tr_Format = new TG_TR_Format();
		String[] values = new String[tg_tr_Format.length()];
		
		// Unrecognized data
		values[tg_tr_Format.getNum(RapidRMAKeywords.R_DATA)] = "RESOURCES";

		// Identifier of the task that acesses the resource
		values[tg_tr_Format.getNum(RapidRMAKeywords.TASKID)] = ""+accessId;

		Resource resource = usage.getResource();
		if (resource!=null) {
			Node node = resource.getNode();
			if (node!=null) {
		
				// Identifier of the node that hosts the resource
				values[tg_tr_Format.getNum(RapidRMAKeywords.NODEID)] = (String) RapidRMAHelper._idTable.get(node);

				// Name of the node that hosts the resource
				values[tg_tr_Format.getNum(RapidRMAKeywords.N_NAME)] = ""+node.getName();
			}
			else {
				Log.warningMessage(Activator.PLUGIN_ID,"The resource "+resource.getName()+" has no node");
			}
					
			// Resource identifier
			values[tg_tr_Format.getNum(RapidRMAKeywords.RES_ID)] = (String) RapidRMAHelper._idTable.get(resource);

			// Resource name
			values[tg_tr_Format.getNum(RapidRMAKeywords.R_NAME)] = resource.getName();

			// Indicates whether the resource is active or not.
			if (resource instanceof CPU) {
				values[tg_tr_Format.getNum(RapidRMAKeywords.A_RES_)] = "true";
			}
			else {
				values[tg_tr_Format.getNum(RapidRMAKeywords.A_RES_)] = "false";
			}
		}
		else {
			Log.warningMessage(Activator.PLUGIN_ID,"A resource usage has no resource");
		}
		
		tg_tr_Format.write(values,writer);
	}

	/**
	 * Serializes the intermediate deadlines into two "xxx-it.tgX" and
	 * "xxx-tr.tgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	private void createResourceUsages() throws Exception {

		// Serializes resources usages into a "xxx-tr.tgX" file

		File file = new File(TG_TR_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file, true);
		PrintWriter writer = new PrintWriter(out,true);

		if (_graph != null) {
			Iterator itTask = _graph.getTask().iterator();
			while (itTask.hasNext()) {
				Task task = (Task) itTask.next();

				Iterator itUsage = task.getResourceusage().iterator();
				while (itUsage.hasNext()) {

					ResourceUsage usage = (ResourceUsage) itUsage.next();
					String accessId = (String) RapidRMAHelper._idTable.get(task);
					writeResourceUsage(usage, accessId, writer);
				}
			}


			writer.close();

			// Serializes resources usages into a "xxx-it.tgX" file
			file = new File(TG_IT_Format.FILENAME);
			out = new FileOutputStream(file, true);
			writer = new PrintWriter(out,true);

			itTask = _graph.getTask().iterator();
			
			while (itTask.hasNext()) {

				int index = 0;
				Map<Object,Integer> indexTable = new HashMap<Object,Integer>();

				Task task = (Task) itTask.next();

//				RapidRMAHashtable resCount = new RapidRMAHashtable("Resource count");

				Iterator itUsage = task.getResourceusage().iterator();
				while (itUsage.hasNext()) {

					ResourceUsage usage = (ResourceUsage) itUsage.next();
				
					TG_IT_Format tg_it_Format = new TG_IT_Format();
					String[] values = new String[tg_it_Format.length()];

					// Writes the index
					values[tg_it_Format.getNum(RapidRMAKeywords.INDEX_)] = ""+index;

					// Writes the task identifier
					values[tg_it_Format.getNum(RapidRMAKeywords.TASKID)] = (String) RapidRMAHelper._idTable.get(task);

					// Dataset
					values[tg_it_Format.getNum(RapidRMAKeywords.DATAID)] = "RESOURCES";

					Resource resource = usage.getResource();
					if (resource !=null) {
						Node node = resource.getNode();
						if (node != null) {
							// Identifier of the node that hosts the resource
							values[tg_it_Format.getNum(RapidRMAKeywords.NODE__)] = (String) RapidRMAHelper._idTable.get(node);
						}
						else {
							Log.warningMessage(Activator.PLUGIN_ID,"The resource "+resource.getName()+" has no node");
						}
					
						// The resource identifier
						values[tg_it_Format.getNum(RapidRMAKeywords.RES_ID)] = (String) RapidRMAHelper._idTable.get(resource);

						// The resource name
						values[tg_it_Format.getNum(RapidRMAKeywords.R_NAME)] = resource.getName();
						
						if (indexTable.containsKey(RapidRMAHelper._idTable.get(resource))) {
							int divIndex = indexTable.get(RapidRMAHelper._idTable.get(resource)).intValue() + 1;
							indexTable.put(RapidRMAHelper._idTable.get(resource), new Integer(divIndex));
							values[tg_it_Format.getNum(RapidRMAKeywords.ID____)] = "DIV" + divIndex;
						} else {
							indexTable.put(RapidRMAHelper._idTable.get(resource), new Integer(0));
							values[tg_it_Format.getNum(RapidRMAKeywords.ID____)] = "DIV0";
						}

/* AFAC Seems to be useless

 						// Counts the usages for a given resource.
						// Defining the several intervals implies several usages.

						if (!resCount.containsKey(resource)) {
							resCount.put(resource, new Integer(0));
						} else {
							int count = ((Integer) resCount.get(resource)).intValue();
							count++;
							resCount.put(resource, new Integer(count));
						}

						int use = ((Integer) resCount.get(resource)).intValue();
						writer.print("DIV" + use);
*/
					}
					else {
						Log.warningMessage(Activator.PLUGIN_ID,"A resource usage has no resource");
					}
				
					values[tg_it_Format.getNum(RapidRMAKeywords.WORK__)] = "0";

					// Writes the time information related to the usage
					values[tg_it_Format.getNum(RapidRMAKeywords.STARTT)] = ""+usage.getStartTime();
					values[tg_it_Format.getNum(RapidRMAKeywords.END_T_)] = ""+usage.getEndTime();
					values[tg_it_Format.getNum(RapidRMAKeywords.ACQ_T_)] = ""+usage.getAcquisitionTime();
					values[tg_it_Format.getNum(RapidRMAKeywords.DEACQT)] = ""+usage.getDeacquisitionTime();
				
					tg_it_Format.write(values,writer);

					index++;
				}
			}
		}
		writer.close();
	}

	/**
	 * Serializes the intermediate deadlines into a "xxx-it.tgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	private void createNonPreemptibleSections() throws Exception {

		File file = new File(TG_IT_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);


		if (_graph != null) {
			Iterator itTask = _graph.getTask().iterator();
			while (itTask.hasNext()) {

				Task task = (Task) itTask.next();
				Iterator itSection = task.getNonPremptible().iterator();
				while (itSection.hasNext()) {

					NonPremptibleSection section = (NonPremptibleSection) itSection.next();
				
					TG_IT_Format tg_it_Format = new TG_IT_Format();
					String[] values = new String[tg_it_Format.length()];
				
					int index = 0;

					// Writes the index
					values[tg_it_Format.getNum(RapidRMAKeywords.INDEX_)] = ""+index;

					// Writes the task identifier
					values[tg_it_Format.getNum(RapidRMAKeywords.TASKID)] = (String) RapidRMAHelper._idTable.get(task);

					// Unrecognized content
					values[tg_it_Format.getNum(RapidRMAKeywords.DATAID)] = ""; 
					values[tg_it_Format.getNum(RapidRMAKeywords.NODE__)] = ""; 
					values[tg_it_Format.getNum(RapidRMAKeywords.RES_ID)] = ""; 
					values[tg_it_Format.getNum(RapidRMAKeywords.R_NAME)] = ""; 
					values[tg_it_Format.getNum(RapidRMAKeywords.ID____)] = "DIV" + index;
					values[tg_it_Format.getNum(RapidRMAKeywords.WORK__)] = "0";

					// Writes the section
					values[tg_it_Format.getNum(RapidRMAKeywords.STARTT)] = ""+section.getStartTime();
					values[tg_it_Format.getNum(RapidRMAKeywords.END_T_)] = ""+section.getEndTime();

					// Unrecognized content
					values[tg_it_Format.getNum(RapidRMAKeywords.ACQ_T_)] = ""+index;
					values[tg_it_Format.getNum(RapidRMAKeywords.DEACQT)] = ""+index + 1;

					tg_it_Format.write(values,writer);
				
					index++;
				}
			}
		}
		writer.close();
	}

	/**
	 * Serializes the intermediate deadlines into a "xxx-id.tgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	private void createIntermediateDeadline() throws Exception {

		File file = new File(TG_IT_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);

		if (_graph != null) {
			int index = 0;
			Iterator itTask = _graph.getTask().iterator();
			while (itTask.hasNext()) {
				Task task = (Task) itTask.next();
				Iterator itDeadline = task.getIntermediate().iterator();
				while (itDeadline.hasNext()) {
					IntermediateDeadline deadline = (IntermediateDeadline) itDeadline.next();
				
					TG_ID_Format tg_id_Format = new TG_ID_Format();
					String[] values = new String[tg_id_Format.length()];

					// Index
					values[tg_id_Format.getNum(RapidRMAKeywords.INDEX_)] = ""+index;

					// Related task
					values[tg_id_Format.getNum(RapidRMAKeywords.TASKID)] = (String) RapidRMAHelper._idTable.get(task);

					// Deadline identifier
					values[tg_id_Format.getNum(RapidRMAKeywords.DEADID)] = "D" + index;

					// Work remaining
					values[tg_id_Format.getNum(RapidRMAKeywords.WORK__)] = ""+deadline.getWorkRemaining();

					// Start time
					values[tg_id_Format.getNum(RapidRMAKeywords.STARTT)] = ""+deadline.getStartTime();

					// End time
					values[tg_id_Format.getNum(RapidRMAKeywords.END_T_)] = ""+deadline.getEndTime();

					// Deadline
					values[tg_id_Format.getNum(RapidRMAKeywords.DEAD__)] = ""+deadline.getIntDeadline();

					// Task relative deadline
					values[tg_id_Format.getNum(RapidRMAKeywords.R_DEAD)] = ""+task.getRelDeadline();

					tg_id_Format.write(values,writer);
				
					index++;
				}
			}
		}
		writer.close();
	}

	/**
	 * Serializes the task dependencies into a "xxx-td.tgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	private void createTaskDependencies() throws Exception {
		File file = new File(TG_TD_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);


		if (_graph != null) {
			// Iterates thru tasks
			Iterator taskIt = _graph.getTask().iterator();
			while (taskIt.hasNext()) {

				// For each task
				Task task = (Task) taskIt.next();
				String taskId = (String) RapidRMAHelper._idTable.get(task);	

				Iterator depIt = task.getDependency().iterator();	
			
				TG_TD_Format tg_td_Format = new TG_TD_Format();
				String[] values = new String[tg_td_Format.length()];
			
				while (depIt.hasNext()) {
					// Write the task identifer
					values[tg_td_Format.getNum(RapidRMAKeywords.PREDID)] = ""+taskId;
					// Write the related task identifiers
					values[tg_td_Format.getNum(RapidRMAKeywords.SUCCID)] = (String) RapidRMAHelper._idTable.get(depIt.next());
					// Unknown feature
					values[tg_td_Format.getNum(RapidRMAKeywords.DEPEND)] = "true";
				
					tg_td_Format.write(values,writer);
				}
			}
		}
		writer.close();
	}

	/**
	 * Serializes the periodic servers into a "xxx-ps.tgX" file.
	 * 
	 * @throws Exception
	 *             if serialization fails
	 */
	private void createPeriodicServer() throws Exception {

		File file = new File(TG_PS_Format.FILENAME);
		FileOutputStream out = new FileOutputStream(file);
		PrintWriter writer = new PrintWriter(out,true);

		if (_graph != null) {
			Iterator serverIt = _graph.getServer().iterator();
			// SD 17/11/07
			//int index = globalIndex;
			int index = _graph.getTask().size();
			while (serverIt.hasNext()) {

				PeriodicServer server = (PeriodicServer) serverIt.next();
			
				TG_PS_Format tg_ps_Format = new TG_PS_Format();
				String[] values = new String[tg_ps_Format.length()];

				// Index
				values[tg_ps_Format.getNum(RapidRMAKeywords.INDEX_)] = ""+index;

				// Server identifier
				values[tg_ps_Format.getNum(RapidRMAKeywords.SERVID)] = "PS" + index;
				RapidRMAHelper._idTable.put(server, "PS" + index);

				// remember that the server shall be registered in the hashtable

				// Server name
				values[tg_ps_Format.getNum(RapidRMAKeywords.S_NAME)] = ""+server.getName();
			
				// Period
				values[tg_ps_Format.getNum(RapidRMAKeywords.PERIOD)] = ""+server.getPeriod();

				// Ex Budget
				values[tg_ps_Format.getNum(RapidRMAKeywords.EXEC_B)] = ""+server.getBudget();

				// Relative Deadline
				values[tg_ps_Format.getNum(RapidRMAKeywords.R_DEAD)] = ""+server.getRelDeadline();

				// Phase
				values[tg_ps_Format.getNum(RapidRMAKeywords.PHASE_)] = ""+server.getPhase();

				// Priority
				values[tg_ps_Format.getNum(RapidRMAKeywords.PRIO__)] = ""+server.getPriority();

				Resource resource = server.getActiveResource();
				if (resource != null) {
					// Active resource
					values[tg_ps_Format.getNum(RapidRMAKeywords.A_RES_)] = (String) RapidRMAHelper._idTable.get(server.getActiveResource());
				}
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"The periodic server "+server.getName()+" has no active resource");
				}

				/*
				 * The fact that a server runs on a CPU is an implicit
				 * kind of resource usage in the RapidRMA tool and meta-model. When
				 * serializing we need to make explicit these usages.
				 */
//				File trFile = new File(TG_TR_Format.FILENAME);
//				FileOutputStream trOut = new FileOutputStream(trFile, true);
//				PrintWriter trWriter = new PrintWriter(trOut,true);
//				ResourceUsage cpuUsage = TaskFactory.eINSTANCE.createResourceUsage();
//				cpuUsage.setResource(server.getActiveResource());
//				writeResourceUsage(cpuUsage, "PS" + index, trWriter);
//				trWriter.close();

				// Server type
				if (server.getType().equals(ServerKind.SPORATIC_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "SS";
				} else if (server.getType().equals(
						ServerKind.SIMPLE_SPORATIC_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "SimSS";
				} else if (server.getType().equals(
					ServerKind.DEADLINE_SPORATIC_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "DdlnSS";
				} else if (server.getType().equals(
					ServerKind.DEADLINE_SIMPLE_SPORATIC_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "DdSmSS";
				} else if (server.getType().equals(
					ServerKind.DEFERRABLE_SERVER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "DefSev";
				} else if (server.getType().equals(
					ServerKind.SIMPLE_SPORADIC_BACKGROUND_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "SmBgSS";
				} else if (server.getType().equals(ServerKind.BACKGROUND_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "Bkgnd";
				} else if (server.getType().equals(
					ServerKind.DEADLINE_SIMPLE_SPORADIC_BACKGROUND_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "DdSmBkSS";
				} else if (server.getType().equals(
					ServerKind.LOCAL_SLACK_STEALER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "LclSlk";
				} else if (server.getType().equals(
					ServerKind.GLOBAL_SLACK_STEALER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "GblSlk";
				} else if (server.getType().equals(
					ServerKind.GREEDY_SLACK_STEALER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "GdySlk";
				} else if (server.getType().equals(
					ServerKind.CONSTANT_UTILIZATION_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "CnstUS";
				} else if (server.getType().equals(ServerKind.PURE_POLLER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "PurPol";
				} else if (server.getType().equals(
					ServerKind.PERSISTENT_POLLER_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "PerPol";
				} else {
					values[tg_ps_Format.getNum(RapidRMAKeywords.S_TYPE)] = "PriXchg";
				}

				// Queue type
				if (server.getQueueing().equals(QueueingKind.FIFO_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.SERV_Q)] = "fifo.queuing";
				} else if (server.getQueueing().equals(QueueingKind.SRPT_LITERAL)) {
					values[tg_ps_Format.getNum(RapidRMAKeywords.SERV_Q)] = "srpt.queuing";
				} else {
					values[tg_ps_Format.getNum(RapidRMAKeywords.SERV_Q)] = "edf.queuing";
				}
			
				// Spare capacity calc required
				values[tg_ps_Format.getNum(RapidRMAKeywords.SCCREQ)] = "false";

				// Coord
				values[tg_ps_Format.getNum(RapidRMAKeywords.XCOORD)] = "0";
				values[tg_ps_Format.getNum(RapidRMAKeywords.YCOORD)] = "0";

				tg_ps_Format.write(values,writer);
			
				index++;
			}
			// SD 17/11/07
			// globalIndex=index;
		}
		out.close();
	}

	
	public void create() {
		try {
			
			/* Creates the periodic server file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Periodic server file");
			createPeriodicServer();
			
			/* Creates the task graph file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Task graph file");
			createTaskGraph();

			/* Creates the intermediate deadlines file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Intermediate deadlines file");
			createIntermediateDeadline();

			/* Creates the task dependencies file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Task dependencies file");
			createTaskDependencies();

			/* Creates the non-preemptible sections file */
			Log.verboseMessage(Activator.PLUGIN_ID,"Non-preemptible sections file");
			createNonPreemptibleSections();

			/* Creates the resource usages files */
			Log.verboseMessage(Activator.PLUGIN_ID,"Resource usages file");
			createResourceUsages();
		}
		catch (Exception exception) {
			Log.errorMessage(Activator.PLUGIN_ID,"Problem in creating TaskHelper",exception);
		}
	}

}
