/** 
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 **/
package com.thalesgroup.marte2rapidrma.popup.actions;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URISyntaxException;
import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.jface.operation.IRunnableWithProgress;

import com.thalesgroup.atl.ExtendedATLTransform;
import com.thalesgroup.atl.marte.MARTEImplementation;
import com.thalesgroup.atl.marte.rsa.MARTEImplementation_RSA;
import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.rapidrma.database.Database;
import com.thalesgroup.marte.rapidrma.export.Activator;
import com.thalesgroup.marte.rapidrma.export.actions.DatabaseHelper;
import com.thalesgroup.marte.rapidrma.export.actions.ResourceHelper;
import com.thalesgroup.marte.rapidrma.export.actions.TaskHelper;

/**
 * Launches the MARTE to RapidRMA transformation by invoking ATL
 */
public class MARTE2RapidRMATransform extends ExtendedATLTransform implements IRunnableWithProgress {	
	
	/**
	 * RapidRMA meta-model constants
	 */
	protected static final String RAPID_RMA_MM_NAME="RapidRMA";
	protected static final String RAPID_RMA_DATABASE_MM_NAME="RapidRMADatabase";
	protected static final String RAPID_RMA_TASK_MM_NAME="RapidRMATask";
	protected static final String RAPID_RMA_RESOURCE_MM_NAME="RapidRMAResource";
	protected static final String RAPID_RMA_MM_URI = "http://www.thalesgroup.com/marte/rapidrma.ecore";
	protected static final String RAPID_RMA_DATABASE_MM_URI = "http://www.thalesgroup.com/marte/rapidrma/database.ecore";
	protected static final String RAPID_RMA_TASK_MM_URI = "http://www.thalesgroup.com/marte/rapidrma/task.ecore";
	protected static final String RAPID_RMA_RESOURCE_MM_URI = "http://www.thalesgroup.com/marte/rapidrma/resource.ecore";
	protected static final String INPUT_MODEL_NAME="IN";
	protected static final String OUTPUT_DATABASE_MODEL_NAME="DATABASE";
	protected static final String OUTPUT_TASK_MODEL_NAME="TASK";
	protected static final String OUTPUT_RESOURCE_MODEL_NAME="RESOURCE";
	
	/**
	 * Input file descriptor
	 */
	private IFile inputFile;
	
	/**
	 * Input file directory
	 */
	private String directory;
	
	/**
	 * Default database file name
	 */
	private String outputDatabaseFile = "DATABASE";
	
	/**
	 * Default task file name
	 */
	private String outputTaskFile = "TASK";
	
	/**
	 * Default resource file name
	 */
	private String outputResourceFile = "RESOURCE";
	
	/**
	 * MARTE2RapidRMATransform default constructor
	 * 
	 * @param transformation URL of the transformation ATL file
	 * @param inputFile Input file to transform
	 * @param directory Directory where to generate files
	 * @throws IOException
	 * @throws CoreException
	 * @throws URISyntaxException
	 */
	public MARTE2RapidRMATransform(URL transformation, IFile inputFile, String directory) 
			throws IOException, CoreException, URISyntaxException {
		super(transformation);
		this.inputFile = inputFile;
		if (directory.charAt(directory.length()-1)==('\\')) {
			this.directory = directory;
		} else {
			this.directory = directory+"\\";
		}
		this.outputDatabaseFile = inputFile.getParent().getLocation().toString()+"\\"+outputDatabaseFile;
		this.outputTaskFile = inputFile.getParent().getLocation().toString()+"\\"+outputTaskFile;
		this.outputResourceFile = inputFile.getParent().getLocation().toString()+"\\"+outputResourceFile;
	}
	
	/**
	 * Triggers the transformation
	 */
	public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
	    monitor.beginTask("Transformation of the model "+inputFile.getName()+" in a RapidRMA format", IProgressMonitor.UNKNOWN);
	    init();
	    monitor.done();
	    if (monitor.isCanceled())
	      throw new InterruptedException("Cancelled");
	}
	
	/**
	 * Initialize the ATL transformation
	 */
	protected void init() {
		try {
			MARTEImplementation marteImpl = new MARTEImplementation_RSA();
			this.enableMARTE(marteImpl);
			this.enableXMLParameters();
			this.enableLOG();
			
			Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA metamodel loading ... "+RAPID_RMA_MM_URI);
			this.addMetamodelByURI(RAPID_RMA_DATABASE_MM_NAME, RAPID_RMA_DATABASE_MM_URI);
			this.addMetamodelByURI(RAPID_RMA_TASK_MM_NAME, RAPID_RMA_TASK_MM_URI);
			this.addMetamodelByURI(RAPID_RMA_RESOURCE_MM_NAME, RAPID_RMA_RESOURCE_MM_URI);
			Log.verboseMessage(Activator.PLUGIN_ID,"Input model loading ... "+inputFile);
			this.addInputModel(INPUT_MODEL_NAME, UML2MM_NAME, inputFile);
			this.createModel(OUTPUT_DATABASE_MODEL_NAME, RAPID_RMA_DATABASE_MM_NAME,true);
			this.createModel(OUTPUT_TASK_MODEL_NAME, RAPID_RMA_TASK_MM_NAME,true);
			this.createModel(OUTPUT_RESOURCE_MODEL_NAME, RAPID_RMA_RESOURCE_MM_NAME,true);
			
			if(Log.isDEBUGEnabled()) this.addXMLParameter("DEBUG", "true");
			if(Log.isVERBOSEEnabled()) this.addXMLParameter("VERBOSE", "true");
			if(Log.isDEMOEnabled()) this.addXMLParameter("DEMO", "true");
			if(Log.isWARNINGEnabled()) this.addXMLParameter("WARNING", "true");
			if(Log.isERROREnabled()) this.addXMLParameter("ERROR", "true");
			
			Log.debugMessage(Activator.PLUGIN_ID,"libs : "+this.libs.toString());
			Log.debugMessage(Activator.PLUGIN_ID,"models : "+this.models.toString());
			Log.debugMessage(Activator.PLUGIN_ID,"params : "+this.params.toString());		
			
			run();
			this.clearXMLParameters();
			removeModel(INPUT_MODEL_NAME);
			Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA model creation");
			Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA database : "+outputDatabaseFile);
			Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA task : "+outputTaskFile);
			Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA resource : "+outputResourceFile);
			this.exportModel(OUTPUT_DATABASE_MODEL_NAME, outputDatabaseFile);
			this.exportModel(OUTPUT_TASK_MODEL_NAME, outputTaskFile);
			this.exportModel(OUTPUT_RESOURCE_MODEL_NAME, outputResourceFile);
			
			ResourceSet resourceSet = new ResourceSetImpl();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(Resource.Factory.Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());
			URI fileURI = URI.createFileURI(outputDatabaseFile);
			Resource resource = resourceSet.getResource(fileURI, true);
			if (resource.getContents().get(0) instanceof Database) {
				Database database = (Database) resource.getContents().get(0);
				Log.verboseMessage(Activator.PLUGIN_ID, "Database : "+database.getName());
				//EM ToDo Appel  la conversion en fichier RapidRMA
				try {
					Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA files generation for database "+database.getName());

					/* Creates the database file */
					DatabaseHelper dbHelper = new DatabaseHelper(database,directory);
					dbHelper.create();

					/* Creates the resource graph file */
					ResourceHelper resHelper = new ResourceHelper(database.getName(), database.getResource());
					resHelper.create();

					/* Creates the task graph file */
					TaskHelper tHelper = new TaskHelper(database.getName(),database.getTask());
					tHelper.create();

				}
				catch (Exception e) {
					Log.errorMessage(Activator.PLUGIN_ID, "Generation of RapidRMA files cancelled", e);
				}
			}
			else {
				Log.errorMessage(Activator.PLUGIN_ID, "The database has not been generated correctly");
			}
						
			removeModel(OUTPUT_DATABASE_MODEL_NAME);
			removeModel(OUTPUT_TASK_MODEL_NAME);
			removeModel(OUTPUT_RESOURCE_MODEL_NAME);
		}
		catch (Exception e) {
			Log.errorMessage(Activator.PLUGIN_ID, "Error while initializing MARTE2RapidRMA ATL transformation", e);
		}
	}
}
