/** 
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 **/

package com.thalesgroup.marte2rapidrma.popup.actions;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URISyntaxException;
import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;

import com.thalesgroup.atl.ExtendedATLProblemsManager;
import com.thalesgroup.atl.ExtendedATLTransform;
import com.thalesgroup.atl.marte.MARTEImplementation;
import com.thalesgroup.atl.marte.rsa.MARTEImplementation_RSA;
import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.rapidrma.export.Activator;

/**
 * Launches the MARTE model check by invoking ATL
 */
public class RapidRMACheckTransform extends ExtendedATLTransform implements IRunnableWithProgress {	
	
	/**
	 * Input model name
	 */
	protected static final String INPUT_MODEL_NAME="IN";
	
	/**
	 * Problem model name (output)
	 */
	protected static final String PROBLEM_MODEL_NAME="problem";
	
	/**
	 * Responsible for analysing the problem and displaying the relevant informatipn
	 */
	protected ExtendedATLProblemsManager manager = new ExtendedATLProblemsManager();
	
	/**
	 * The selected input file
	 */
	private IFile inputFile;

	/**
	 * RapidRMACheckTransform default constructor
	 * 
	 * @param transformation the transformation URL
	 * @param inputFile the input file
	 * @throws IOException if an error occurs when reading the file
	 * @throws CoreException if an Eclipse-related error occurs
	 * @throws URISyntaxException if the provided transformation URL is malformed
	 */
	public RapidRMACheckTransform(URL transformation, IFile inputFile) 
			throws IOException, CoreException, URISyntaxException {
		super(transformation);
		this.inputFile = inputFile;
	}
	
	/**
	 * Executes the transformation
	 */
	public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
	    monitor.beginTask("Checking of the model "+inputFile.getName()+" for transformation in a RapidRMA format", IProgressMonitor.UNKNOWN);
	    init();
	    monitor.done();
	    if (monitor.isCanceled())
	      throw new InterruptedException("Cancelled");
	}
	
	/**
	 * Initializes the transformation
	 */
	protected void init() {
		try {
			MARTEImplementation marteImpl = new MARTEImplementation_RSA();
			this.enableMARTE(marteImpl);
			this.enableXMLParameters();
			this.enableLOG();
			this.enablePROBLEM();
			
			manager.cleanProblems(inputFile);
			
			Log.verboseMessage(Activator.PLUGIN_ID,"Input model loading ... "+inputFile);
			this.addInputModel(INPUT_MODEL_NAME, UML2MM_NAME, inputFile);
			
			this.createProblemModel(PROBLEM_MODEL_NAME);
			
			if(Log.isDEBUGEnabled()) this.addXMLParameter("DEBUG", "true");
			if(Log.isVERBOSEEnabled()) this.addXMLParameter("VERBOSE", "true");
			if(Log.isDEMOEnabled()) this.addXMLParameter("DEMO", "true");
			if(Log.isWARNINGEnabled()) this.addXMLParameter("WARNING", "true");
			if(Log.isERROREnabled()) this.addXMLParameter("ERROR", "true");
			
			Log.debugMessage(Activator.PLUGIN_ID,"libs : "+this.libs.toString());
			Log.debugMessage(Activator.PLUGIN_ID,"models : "+this.models.toString());
			Log.debugMessage(Activator.PLUGIN_ID,"params : "+this.params.toString());		
			
			run();
			this.clearXMLParameters();
			
			manager.importModel(getResourceModel(PROBLEM_MODEL_NAME));
			manager.signalProblems(inputFile);
			
			removeModel(INPUT_MODEL_NAME);
			removeModel(PROBLEM_MODEL_NAME);
		}
		catch (Exception e) {
			Log.errorMessage(Activator.PLUGIN_ID, "Error while initializing MARTE2RapidRMA ATL transformation", e);
		}
	}
}
