/** 
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 **/

package com.thalesgroup.marte2rapidrma.popup.actions;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URISyntaxException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.rapidrma.database.Database;
import com.thalesgroup.marte.rapidrma.export.Activator;
import com.thalesgroup.marte.rapidrma.export.actions.DatabaseHelper;
import com.thalesgroup.marte.rapidrma.export.actions.ResourceHelper;
import com.thalesgroup.marte.rapidrma.export.actions.TaskHelper;

/**
 * Generates RapidRMA files from a related RapidRMA model described with EMF
 * @author demathieu
 */
public class RapidRMAFilesGeneration implements IRunnableWithProgress, IObjectActionDelegate {
	
	/**
	 * The selected input file
	 */
	private IFile inputFile;
	
	/**
	 * The target output directory
	 */
	String directory;
	
	/**
	 * RapidRMAFilesGeneration default constructor
	 * 
	 * @param transformation URL of the transformation ATL file
	 * @param inputFile Input file to transform
	 * @param directory Directory where to generate files
	 * @throws IOException if an IO error occurs
	 * @throws CoreException if an Eclipse-related error occurs
	 * @throws URISyntaxException if the provided URL is malformed
	 */
	public RapidRMAFilesGeneration() 
			throws IOException, CoreException, URISyntaxException {
	}
	
	/**
	 * Starts the generation of RapidRMA files
	 */
	public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
	    monitor.beginTask("Generation of RapidRMA Files for "+inputFile.getLocation().toString(), IProgressMonitor.UNKNOWN);
	    init();
	    monitor.done();
	    if (monitor.isCanceled())
	      throw new InterruptedException("Cancelled");
	}
	
	/**
	 * Sets the active workbench part
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
	}
	
	/**
	 * Executes the generation program
	 */
	public void run(IAction action) {
		init();
	}
	
	/**
	 * Initializes the generation
	 */
	protected void init() {
		try {
					
			ResourceSet resourceSet = new ResourceSetImpl();
			resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put(Resource.Factory.Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());
			URI fileURI = URI.createFileURI(inputFile.getLocation().toString());
			Resource resource = resourceSet.getResource(fileURI, true);
			if (resource.getContents().get(0) instanceof Database) {
				Database database = (Database) resource.getContents().get(0);
				Log.verboseMessage(Activator.PLUGIN_ID, "Database : "+database.getName());
				//EM ToDo Appel  la conversion en fichier RapidRMA
				try {
					Log.verboseMessage(Activator.PLUGIN_ID,"RapidRMA files generation for database "+database.getName());

					/* Creates the database file */
					DatabaseHelper dbHelper = new DatabaseHelper(database,directory);
					dbHelper.create();

					/* Creates the resource graph file */
					ResourceHelper resHelper = new ResourceHelper(database.getName(), database.getResource());
					resHelper.create();

					/* Creates the task graph file */
					TaskHelper tHelper = new TaskHelper(database.getName(),database.getTask());
					tHelper.create();
					
					/* Refresh*/
					inputFile.getParent().refreshLocal(1, null);
				}
				catch (Exception e) {
					Log.errorMessage(Activator.PLUGIN_ID, "Generation of RapidRMA files cancelled", e);
				}
			}
			else {
				Log.errorMessage(Activator.PLUGIN_ID, "The database has not been generated correctly");
			}
		}
		catch (Exception e) {
			Log.errorMessage(Activator.PLUGIN_ID, "Error while initializing RapidRMA Files generation", e);
		}
	}

	/**
	 * Callback triggered when the action gets changed
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		try {
			inputFile = (IFile) ((TreeSelection) selection).getFirstElement();
			directory = inputFile.getLocation().removeLastSegments(1).toString()+"\\";
		}
		catch (Exception e) {
			Log.errorMessage(Activator.PLUGIN_ID, "Error in file selection", e);
		}
	}
}
