/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>
 *
 * $Id: VariablesCollection.java,v 1.6 2007/10/24 07:06:56 fnizou Exp $
 */
package com.thalesgroup.nfp.rsa.facade;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;

import VSL.ExpressionContext;
import VSL.VSLFactory;
import VSL.Variable;
import VSL.VariableDirectionKind;

import com.cea.nfp.parsers.modelgenerator.VSLFullParser;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;
import com.ibm.xtools.emf.index.search.IIndexSearchManager;
import com.ibm.xtools.emf.index.search.IndexContext;

/**
 * 
 * @author T0081227 Francois NIZOU - 20 aot 07
 * 
 */
public class VariablesCollection {

	protected Model model;

	protected RSAModelFacade facade;

	protected HashMap<String, ArrayList<Variable>> variables = null;

	protected ArrayList<Variable> flatVariables = null;

	protected Element current;

	public VariablesCollection(Model model, RSAModelFacade facade,
			Element current) {
		this.model = model;
		this.facade = facade;
		this.current = current;
	}

	public void init() {
		variables = new HashMap<String, ArrayList<Variable>>();
		flatVariables = new ArrayList<Variable>();

		flatVariables = getVariableFromOpaque();
		flatVariables.addAll(getVariablesFromProperty());
		flatVariables.addAll(getVariablesFromVarProperty());
		flatVariables.addAll(getVariablesFromStereotypeAttribute());

		for (Variable var : flatVariables) {
			String name = var.getName();
		
			if (!variables.containsKey(name))
				variables.put(name, new ArrayList<Variable>());
			variables.get(name).add(var);

			ExpressionContext context = var.getContext();
			if (context != null) {
				String qName = context.getName() + "." + name;
				if (!variables.containsKey(qName))
					variables.put(qName, new ArrayList<Variable>());
				variables.get(qName).add(var);
			}
		}

	}

	private ArrayList<Variable> getVariablesFromVarProperty() {
		ArrayList<Variable> result = new ArrayList<Variable>();

		ArrayList<Property> properties = facade.getProperties();

		ArrayList<Resource> ressources = new ArrayList<Resource>();
		ressources.add(model.eResource());

		IndexContext indexContext = new IndexContext(model.eResource()
				.getResourceSet(), ressources);
		indexContext.getOptions().put(IndexContext.RESOLVE_PROXIES,
				Boolean.TRUE);
		Collection props = null;

		try {
			props = IIndexSearchManager.INSTANCE.findEObjects(indexContext,
					"*", false, null, UMLPackage.eINSTANCE.getProperty(),
					new NullProgressMonitor());
		} catch (Exception e) {

		}
		
		for (Object o : props) {
			Property property = (Property)o;
			EList appliedStereotypes = property.getAppliedStereotypes();
			for (Object object : appliedStereotypes) {
				Stereotype s = (Stereotype) object;
				if (s.getName().equals(MarteCst.VSL.STEREOTYPE_VAR)) {
					Variable variable = VSLFactory.eINSTANCE.createVariable();
					variable.setName(property.getName());

					
					EnumerationLiteral value = (EnumerationLiteral) property
							.getValue(s, "dir");
					if (value.getName().equals("in"))
						variable.setDirection(VariableDirectionKind.IN_LITERAL);
					else if (value.getName().equals("out"))
						variable
								.setDirection(VariableDirectionKind.OUT_LITERAL);
					else if (value.getName().equals("inout"))
						variable
								.setDirection(VariableDirectionKind.INOUT_LITERAL);

					String ctxName = facade.getContextFromElement(property);
					ExpressionContext ctx = VSLFactory.eINSTANCE
							.createExpressionContext();
					ctx.setName(ctxName);
					variable.setContext(ctx);

					DataType type = facade.typeof(property);
					variable.setDatatype(type);
					variable.setType(type);
					variable.setDataTypeName(type.getName());

					String vslValue = property.getDefault();

					try {
						ValueSpecification vsl = VSLFullParser.parse(vslValue,
								type, facade);
						variable.setInitExpression(vsl);
					} catch (Exception e) {
						variable.setInitExpression(UMLFactory.eINSTANCE
								.createLiteralNull());
					}
					result.add(variable);
				}
			}

		}

		return result;

	}

	private ArrayList<Variable> getVariablesFromProperty() {
		ArrayList<Variable> result = new ArrayList<Variable>();

		ArrayList<Property> properties = facade.getProperties();

		for (Property property : properties) {
			DataType type = facade.typeof(property);
			if (type == null)
				continue;
			if (property.getDefault() == null)
				continue;
			ArrayList<Variable> vars = findVariablesInStr(property, property
					.getDefault());
			result.addAll(vars);
		}

		return result;
	}

	/**
	 * 
	 * @return All the VSL variables of the Model.
	 */
	ArrayList<Variable> getVariables() {
		if (variables == null)
			init();
		return flatVariables;

	}

	/**
	 * 
	 * @param prefix
	 * @return all the VSL variables of the Model Starting with startwith
	 */
	ArrayList<Variable> getVariablesStartWith(String prefix) {
		if (variables == null)
			init();
		ArrayList<Variable> results = new ArrayList<Variable>();

		Set<String> keySet = variables.keySet();
		for (String key : keySet) {
			if (!key.startsWith(prefix))
				continue;
			ArrayList<Variable> arrayList = variables.get(key);
			for (Variable variable : arrayList) {
				if (!variable.getName().equals(key))
					results.add(variable);
				else {
					String ctxName = facade.getContextFromElement(current);
					if (ctxName.startsWith(variable.getContext().getName()))
						results.add(variable);
				}

			}
		}
		return results;

	}

	/**
	 * 
	 * @param namespace
	 * @param name
	 * @return all Variables called namespace.name
	 */
	ArrayList<Variable> getVariablesByName(String namespace, String name) {
		if (variables == null)
			init();
		String contextFromElement = facade.getContextFromElement(current);
		if (contextFromElement == null)
			contextFromElement = "";

		String fqn = (namespace == null || namespace.length() == 0) ? (contextFromElement
				.length() == 0) ? name : contextFromElement + "." + name
				: namespace + "." + name;

		if (variables.containsKey(fqn))
			return variables.get(fqn);
		else
			return new ArrayList<Variable>();

	}

	protected ArrayList<Variable> getVariableFromOpaque() {
		ArrayList<Variable> result = new ArrayList<Variable>();
		ResourceSet resourceSet = model.eResource().getResourceSet();
		// EList resources = resourceSet.getResources();

		ArrayList<Resource> resources = new ArrayList<Resource>();
		resources.add(model.eResource());
		IndexContext indexContext = new IndexContext(resourceSet, resources);
		// indexContext.getOptions().put(IndexContext.SEARCH_UNLOADED_RESOURCES,
		// Boolean.TRUE);
		indexContext.getOptions().put(IndexContext.RESOLVE_PROXIES,
				Boolean.TRUE);
		Collection oes = null;
		try {
			oes = IIndexSearchManager.INSTANCE.findEObjects(indexContext,
					"*$*:*", false, null, UMLPackage.eINSTANCE
							.getOpaqueExpression(), new NullProgressMonitor());
			for (Iterator iter = oes.iterator(); iter.hasNext();) {
				OpaqueExpression oe = (OpaqueExpression) iter.next();
				EList languages = oe.getLanguages();
				String txt = "";
				for (int i = 0; i < languages.size(); i++) {
					if (languages.get(i).equals(VSL.VSL_LANGUAGE)) {
						txt = oe.getBodies().get(i).toString();
					}
				}
				ArrayList<Variable> var = findVariablesInStr(oe, txt);
				result.addAll(var);
			}
		} catch (Exception e) {
			return null;
		}

		return result;
	}

	protected ArrayList<Variable> findVariablesInStr(Element containing,
			String txt) {
		ArrayList<Variable> result = new ArrayList<Variable>();
		String ctxName = facade.getContextFromElement(containing);
		ExpressionContext ctx = VSLFactory.eINSTANCE.createExpressionContext();
		ctx.setName(ctxName);
		String tmp = txt.toString();

		int indexOfDollar = tmp.indexOf("$");
		while (indexOfDollar != -1) {
			VariableDirectionKind dir = null;

			String dirstr = tmp.substring(indexOfDollar - 2, indexOfDollar);
			dir = (dirstr.indexOf("in") != -1) ? VariableDirectionKind.IN_LITERAL
					: (dirstr.indexOf("out") != -1) ? VariableDirectionKind.OUT_LITERAL
							: VariableDirectionKind.INOUT_LITERAL;

			String name = tmp.substring(indexOfDollar + 1, tmp.indexOf(":"));
			tmp = tmp.substring(tmp.indexOf(":", indexOfDollar) + 1, tmp
					.length());
			int indexOfDatatypeEnd = 0;
			while (indexOfDatatypeEnd < tmp.length()
					&& Character.isLetter(tmp.charAt(indexOfDatatypeEnd))) {
				indexOfDatatypeEnd++;
			}

			String datatypeName = tmp.substring(0, indexOfDatatypeEnd).trim();
			tmp = tmp.substring(indexOfDatatypeEnd, tmp.length());

			Variable v = VSLFactory.eINSTANCE.createVariable();
			v.setName(name);
			DataType datatype = facade.getDataTypesByName(datatypeName).get(0);
			v.setType(datatype);
			v.setDatatype(datatype);
			v.setDataTypeName(datatypeName);
			v.setDirection(dir);
			v.setContext(ctx);
			result.add(v);
			indexOfDollar = tmp.indexOf("$");
		}

		return result;
	}

	public ArrayList<Variable> getVariablesFromStereotypeAttribute() {
		ArrayList<Variable> result = new ArrayList<Variable>();

		EList ownedElements = model.allOwnedElements();

		for (Object object : ownedElements) {
			Element elt = (Element) object;
			String contextName = facade.getContextFromElement(elt.getOwner());
			ExpressionContext ec = VSLFactory.eINSTANCE
					.createExpressionContext();
			ec.setName(contextName);

			EList stereotypes = elt.getAppliedStereotypes();
			for (Object object2 : stereotypes) {
				Stereotype stereotype = (Stereotype) object2;
				EList allAttributes = stereotype.getAllAttributes();
				for (Object object3 : allAttributes) {
					Property property = (Property) object3;
					Object value = (Object) elt.getValue(stereotype, property
							.getName());

					if (value instanceof EObject) {
						EObject eobj = (EObject) value;
						result.addAll(findVariableInTaggedValue(elt, eobj, ec));
					} else if (value instanceof EList) {
						EList elist = (EList) value;
						result.addAll(findVariableInMultiValuedTaggedValue(elt,
								elist, ec));
					}

				}
			}
		}

		return result;
	}

	/**
	 * 
	 * @param container
	 * @param value
	 * @param ec
	 * @return
	 */
	protected ArrayList<Variable> findVariableInTaggedValue(Element container,
			EObject value, ExpressionContext ec) {
		ArrayList<Variable> result = new ArrayList<Variable>();
		if (value == null)
			return result;
		EClass eclass = value.eClass();
		EStructuralFeature sf = eclass.getEStructuralFeature("value");
		if (sf == null)
			return result;
		Object val = value.eGet(sf);
		if (val instanceof String) {
			String vsl = (String) val;
			ArrayList<Variable> variables = findVariablesInStr(container, vsl);
			result.addAll(variables);
		}
		return result;
	}

	/**
	 * 
	 * @param container
	 * @param value
	 * @param ec
	 * @return
	 */
	protected ArrayList<Variable> findVariableInMultiValuedTaggedValue(
			Element container, EList value, ExpressionContext ec) {
		ArrayList<Variable> result = new ArrayList<Variable>();
		for (Object object : value) {
			if (object != null)
				result.addAll(findVariableInTaggedValue(container,
						(EObject) object, ec));
		}
		return result;
	}
}
