/**
 * <copyright>
 * Thales ATL (Copyright (c) THALES 2007 All rights reserved)
 * is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published
 * in http://www.eclipse.org/legal/epl-v10.html
 * Thales ATL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 * </copyright>
 *
 * $Id: ATLTransform.java,v 1.4 2008/06/16 09:40:14 emaes Exp $
 */
package com.thalesgroup.atl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.atl.eclipse.engine.AtlEMFModelHandler;
import org.atl.eclipse.engine.AtlLauncher;
import org.atl.engine.repositories.emf4atl.ASMEMFModel;
import org.atl.engine.vm.ClassNativeOperation;
import org.atl.engine.vm.Operation;
import org.atl.engine.vm.StackFrame;
import org.atl.engine.vm.nativelib.ASMModel;
import org.atl.engine.vm.nativelib.ASMOclAny;
import org.atl.engine.vm.nativelib.ASMOclType;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;

import com.thalesgroup.java.log.Log;

/**
 * This library provides an easy-to-use customizable interface for ATL transformations
 * @author Nicolas Vienne
 * @version 0.0.3, 09/2007
 * 
 */

public abstract class ATLTransform {
	
	// PlugIn ID
	public static String PluginID = "com.thalesgroup.atl";
	
	// EMF model handler
	protected AtlEMFModelHandler emfamh = null;

	// Transformation
	protected Map params = Collections.EMPTY_MAP; // Parameters

	//protected Map<String, String> libs = new HashMap<String, String>(); // Libraries
	protected Map<String, URL> libs = new HashMap<String, URL>(); // Libraries

	// For easy handling each model is associated to a String,
	// the same that is used in the ATL transformation.
	protected Map<String, ASMModel> models = new HashMap<String, ASMModel>(); // Models

	protected URL atl_transformation;
	
	/*
	 * Constructeur
	 * Force the user to specify a transformation
	 */
	protected ATLTransform(URL transformation) {
		this.atl_transformation = transformation;
	}
	
	/*
	 * Exceptions
	 */
	
	public class ATLTransformException extends Exception {
		private static final long serialVersionUID = 1L;

		public ATLTransformException(String details) {
			super(details);
		}
	}
	
	public class NotLoadedException extends ATLTransformException {
		private static final long serialVersionUID = 1L;

		public NotLoadedException(String details) {
			super(details);
		}
	}
	
	public class UndefinedException extends ATLTransformException {
		private static final long serialVersionUID = 1L;

		public UndefinedException(String details) {
			super(details);
		}
	}
	
	/*
	 * Libraries management
	 */
	protected void addLibrary(String name, URL library) {
		libs.put(name, library);
	}

	protected URL removeLibrary(String name) {
		return libs.remove(name);
	}
	protected void removeAllLibraries() {
		libs.clear();
	}
	
	
	/*
	 * Parameters management
	 */
	protected Map getParameters() {

		return params;
	}

	protected void setParameters(Map params) {

		this.params = params;
	}

	/*
	 * Init the model handler
	 */
	protected void initEMF() {
		if (emfamh == null) { // if EMF is not initialized
			// Initialize EMF model handler
			emfamh = (AtlEMFModelHandler) AtlEMFModelHandler.getDefault(AtlEMFModelHandler.AMH_EMF);
		}
	}

	/*
	 * Load models functions
	 */
	protected ASMModel loadModel(String name, ASMModel metamodel,
			InputStream inputstream) {

		initEMF();
		return emfamh.loadModel(name, metamodel, inputstream);
	}
	
	protected ASMModel loadModel(String name, ASMModel metamodel,
			String uri) {

		initEMF();
		return emfamh.loadModel(name, metamodel, uri);
	}
	

	/*
	 * 
	 */
	protected void addMetamodelByURI(String name, String uri) throws IOException {
		initEMF();
		ASMModel m = loadModel(name, emfamh.getMof(), "uri:"+uri);
		models.put(name, m);
	}
	
	protected void addInputModel(String name, String metamodel, URL file)
			throws IOException {
		ASMModel m = loadModel(name, models.get(metamodel), file.openStream());
		models.put(name, m);
	}

	protected void addMetamodel(String name, URL file) throws IOException {
		initEMF();
		ASMModel m = loadModel(name, emfamh.getMof(), file.openStream());
		models.put(name, m);
	}

	protected void addInputModel(String name, String metamodel, String file)
			throws IOException {
		ASMModel m = loadModel(name, models.get(metamodel),	new FileInputStream(file));
		models.put(name, m);
	}

	protected void addMetamodel(String name, String file) throws IOException {
		initEMF();
		ASMModel m = loadModel(name, emfamh.getMof(), new FileInputStream(file));
		models.put(name, m);
	}

	protected void addInputModel(String name, String metamodel, IFile file)
			throws IOException, CoreException {
		ASMModel m = loadModel(name, models.get(metamodel), file.getContents());
		models.put(name, m);
	}

	protected void addMetamodel(String name, IFile file) throws IOException,
			CoreException {
		initEMF();
		ASMModel m = loadModel(name, emfamh.getMof(), file.getContents());
		models.put(name, m);
	}

	protected void addInputModel(String name, String metamodel,
			InputStream inputstream) throws IOException {	
		ASMModel m = loadModel(name, models.get(metamodel),inputstream);
		models.put(name, m);
	}

	protected void addMetamodel(String name, InputStream inputstream)
			throws IOException {
		initEMF();
		ASMModel m = loadModel(name, emfamh.getMof(), inputstream);
		models.put(name, m);
	}

	protected void createModel(String name, String metamodel, boolean interModelReference) {
		initEMF();
		ASMModel outmodel = emfamh.newModel(name, models.get(metamodel)); // Output model
		if (interModelReference) {
			if (outmodel instanceof ASMEMFModel) {
				ASMEMFModel outEMFModel= (ASMEMFModel) outmodel;
				outEMFModel.setCheckSameModel(false);
			}
		}
		models.put(name, outmodel);
	}
	protected void createModel(String name, String metamodel) {
		createModel(name, metamodel, false);
	}

	/*
	 * Model management functions
	 */

	protected Resource getResourceModel(String name) {
		ResourceSet resourceSet = ASMEMFModel.getResourceSet();
		return resourceSet.getResource(URI.createURI(name), false);
	}
	
	protected void renameModel(String old_name, String new_name) {
		Resource res = getResourceModel(old_name);
		res.setURI(URI.createURI(new_name));
		models.put(new_name, models.remove(old_name));
	}
	/*
	 * Movel removal functions
	 * The model is saved from Resources so multiple runs aim to create
	 * multiple resources with the same same. Since this name is used 
	 * to get the model, to ensure we get the right model, only one 
	 * resource with this name must be available.
	 */
	
	protected void removeModel(String name) {
		models.remove(name);
		ResourceSet resourceSet = ASMEMFModel.getResourceSet();
		Resource res =  resourceSet.getResource(URI.createURI(name), false);
		res.unload();
		resourceSet.getResources().remove(res);	
	}

	protected void removeAllModels() {
		for(String n : models.keySet()) {
			removeModel(n);
		}
	}

	/*
	 * Model export and save functions
	 */
	protected void exportModel(String name, OutputStream outputstream)
			throws IOException {
		Resource res = getResourceModel(name);
		if (res instanceof XMIResourceImpl) {
			XMIResourceImpl xri = (XMIResourceImpl) res;
			xri.save(outputstream, null);
			outputstream.flush();
		} else {
			throw new IllegalArgumentException(name);
		}
	}

	protected void exportModel(String name, IFile file) throws CoreException,
			IOException {
		String smodel = exportModelToString(name);
		exportStringToIFile(smodel, file);
	}

	protected void exportModel(String name, String file) throws IOException {
		FileOutputStream fos = new FileOutputStream(file);
		exportModel(name, fos);
		fos.close();
	}

	protected String exportModelToString(String name) throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		exportModel(name, baos);
		baos.close();
		return baos.toString();
	}

	protected void exportStringToIFile(String s, IFile file) throws CoreException {
		if(file.exists()) {
			file.setContents(new ByteArrayInputStream(s.getBytes()), true, false,
				new NullProgressMonitor());
		} else {
			file.create(new ByteArrayInputStream(s.getBytes()), true, new NullProgressMonitor());
		}
				
	}

	
	/**
	 * Runs the transformation
	 */
	protected void run() {
		if(atl_transformation == null) throw new NullPointerException();
		AtlLauncher.getDefault().launch(atl_transformation, libs, models, params);
	}
	protected void run(URL transformation) {
		AtlLauncher.getDefault().launch(transformation, libs, models, params);
	}
	
	/*
	 * Support for custom operations ...
	 */
	/**
	 * 
	 * @param cl
	 * @param targetClass
	 * @return true if cl is kind of targetClass, false otherwise
	 */
	private static boolean isKindOf(Class cl, Class targetClass) {
		if(cl == targetClass) return true;
		else if(cl.getSuperclass() != null) return isKindOf(cl.getSuperclass(), targetClass);
		else return false;
	}
	/**
	 * Find the first ATL compliant operation named name in class cl.
	 * This operation is the first function that is modified
	 * public static, has at least two parameters, has its first parameter
	 * of type StackFrame, has all of its parameters kind of ASMOclAny, has its
	 * return type kind of ASMOclAny
	 * @param cl
	 * @param name
	 */
	protected static Method findATLCompliantMethod(Class cl, String name) {
		
		Method[] methods = cl.getDeclaredMethods();
		
		for (Method method : methods) {
			
			if(method.getName().equals(name)) {
				
				int mods = method.getModifiers();
				if(!Modifier.isPublic(mods)) continue;
				if(!Modifier.isStatic(mods)) continue;
				
				Class[] parameters = method.getParameterTypes();
				
				if(parameters.length >= 2) {
					
					boolean paramsareasmoclany = true;
					for (Class class1 : parameters) {
						paramsareasmoclany &= isKindOf(class1, ASMOclAny.class);
					}
					if(!paramsareasmoclany) continue;
					
					if(parameters[0].equals(StackFrame.class)
							&& isKindOf(method.getReturnType(), ASMOclAny.class)) {
							
						return method;
					}
				}
			}
		}
		return null;
	}
	/**
	 * Adds the target operation to the transformation virtual machine.
	 * The context is deduced from the operation.
	 * @param cl
	 * @param name
	 */
	public static void addCustomOperationToVM(Class cl, String name) {
		Method method = findATLCompliantMethod(cl, name);
		if(method != null) {
			Operation op = new ClassNativeOperation(method);
			Class[] parameters = method.getParameterTypes();
			Field t;
			try {
				t = parameters[1].getField("myType");
				addCustomOperationToVM((ASMOclType)t.get(parameters[1]), op);
			} catch (Exception e) {
				Log.errorMessage(ATLTransform.PluginID, "Error in adding an operation to the ATL VM", e);
			}
		} else {
			throw new IllegalArgumentException(cl.getSimpleName() + "." + name);
		}
	}
	/**
	 * Adds the target operation named name of the class cl to the context
	 * @param context
	 * @param cl
	 * @param name
	 */
	protected static void addCustomOperationToVM(ASMOclType context, Class cl, String name) {
		Method method = findATLCompliantMethod(cl, name);
		if(method != null) {
			addCustomOperationToVM(context, new ClassNativeOperation(method));
		} else {
			throw new IllegalArgumentException(cl.getSimpleName() + "." + name);
		}
	}
	/**
	 * Adds the target operation to the transformation virtual machine.
	 * @param context
	 * @param op
	 */
	@SuppressWarnings("unchecked")
	protected static void addCustomOperationToVM(ASMOclType context, Operation op) {
		//System.out.println("[ATL_VM] Adding operation "+op.getName()+" ...");
		((Map)ASMOclType.getVMOperations().get(context)).put(op.getName(), op);		
	}
	
}