/**
 * <copyright>
 * Thales ATL (Copyright (c) THALES 2007 All rights reserved)
 * is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published
 * in http://www.eclipse.org/legal/epl-v10.html
 * Thales ATL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 * </copyright>
 *
 * $Id: ExtendedATLProblemsManager.java,v 1.3 2008/06/16 09:40:14 emaes Exp $
 */
package com.thalesgroup.atl;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * Manager for ATL Problems
 * - transforms a PROBLEM model into a list of ATLProblem
 * - manages the markers of theses problems for user notification
 * @author Nicolas Vienne
 * @version 0.0.3, 09/2007
 */
public class ExtendedATLProblemsManager {
	
	protected static Map<IResource, List<IMarker>> markers = new HashMap<IResource, List<IMarker>>();
	
	protected int critic_count = 0;
	protected int error_count = 0;
	protected int warning_count = 0;
	
	protected ArrayList<ExtendedATLProblem> problems = new ArrayList<ExtendedATLProblem>();
	
	public void importModel(Resource model) {

		problems.clear();
		critic_count = 0;
		error_count = 0;
		warning_count = 0;
		
		for(Object o : model.getContents()) {
			if (o instanceof DynamicEObjectImpl) {
				DynamicEObjectImpl dob = (DynamicEObjectImpl) o;
				if(dob.eClass().getName().toString().equals("Problem")) {
					ExtendedATLProblem p = new ExtendedATLProblem(dob);
					
					// Avoid duplicate data		
					if(!problems.contains(p)) {
						problems.add(p);
					}
				}
			}
		}
		
		// Computes the number of problems
		for(ExtendedATLProblem p : problems) {
			switch (p.getSeverity()) {
			case critic:
				critic_count++;
				break;
			case error:
				error_count++;
				break;
			case warning:
				warning_count++;
				break;
			default:
				break;
			}
		}
	}
	

	public List<ExtendedATLProblem> getProblems() {
		return problems;
		
	}

	protected static void appendMarker(IMarker marker, IResource resource) throws CoreException {
		if(!markers.containsKey(resource)) {
			markers.put(resource, new ArrayList<IMarker>());
		}
		markers.get(resource).add(marker);
	}
	
	public void signalProblems(IResource resource) throws CoreException {
		for(ExtendedATLProblem p : problems) {
			IMarker marker = p.createMarker(resource);
			appendMarker(marker, resource);
		}
	}

	public void cleanProblems(IResource resource) throws CoreException {
		cleanProblems(resource, true);
	}
	
	public static void cleanProblems(IResource resource, Boolean flush_all) throws CoreException {
		if(flush_all) {
			IMarker[] problems = null;
			int depth = IResource.DEPTH_INFINITE;
			problems = resource.findMarkers(IMarker.PROBLEM, true, depth);
			for(IMarker m : problems) {
				m.delete();
			}
		} else {
			if(markers.containsKey(resource)) {
				List<IMarker> ml = markers.get(resource);
				for(IMarker m : ml) {
					if(m.exists()) {
						m.delete();
					}
				}
				ml.clear();
			}
		}
	}


	public int getCritic_count() {
		return critic_count;
	}


	public int getError_count() {
		return error_count;
	}


	public int getWarning_count() {
		return warning_count;
	}
}
