/**
 * <copyright>
 * Thales ATL (Copyright (c) THALES 2007 All rights reserved)
 * is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published
 * in http://www.eclipse.org/legal/epl-v10.html
 * Thales MARTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 * </copyright>
 *
 * $Id: ExtendedATLTransform.java,v 1.3 2007/11/21 13:59:45 emaes Exp $
 */
package com.thalesgroup.atl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.atl.engine.extractors.xml.XMLExtractor;
import org.atl.engine.injectors.xml.XMLInjector;
import org.atl.engine.vm.nativelib.ASMModel;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;

import com.thalesgroup.atl.exception.ATLException;
import com.thalesgroup.atl.log.LOGATLFacilities;
import com.thalesgroup.atl.marte.MARTEATLFacilities;
import com.thalesgroup.atl.marte.MARTEFacilities;
import com.thalesgroup.atl.marte.MARTEImplementation;
import com.thalesgroup.java.log.Log;

 /**
 * This extension of ATLTransform adds user-friendly tools for
 * transforming XML, UML or MARTE models, to pass parameters to
 * the transformation, to check a model via the PROBLEM metamodel
 * and to display debugging data.
 * @author Nicolas Vienne
 * @author Eric MAES
 * @version 0.0.3, 13/09/2007
 * 
 */
public class ExtendedATLTransform extends ATLTransform {
	
	/*
	 * Flags for automatic loading
	 */
	public static final int LOG = 0;
	public static final int UML = 1;
	public static final int MARTE = 2;
	public static final int XML = 3;
	public static final int PROBLEM = 4;
	public static final int XMLPARAMETERS = 5;
	/*
	 * LOG Variables
	 */
	protected static final String LOG_LIBRARY_NAME = "LOG";
	protected static final String LOG_LIBRARY_PATH = "/libraries/LOG.asm";
	/*
	 * UML Variables
	 */
	public static final String UML2MM_NAME="UML";
	public static final String UML2MM_URI="http://www.eclipse.org/uml2/2.0.0/UML";
	protected static final String UML2_LIBRARY_NAME = "UML2";
	protected static final String UML2_LIBRARY_PATH = "/libraries/UML2.asm";
	
	/*
	 * MARTE Variables
	 */
	protected static MARTEImplementation marteImplementation;
	protected static final String MARTE_LIBRARY_NAME = "MARTE";
	
	/*
	 * XML Variables
	 */
	public static final String XMLMM_NAME="XML";
	public static final String XMLMM_BUNDLE="org.eclipse.am3.core";
	public static final String XMLMM_PATH="/resources/XML/XML.ecore";
	
	/*
	 * PROBLEMS Variables
	 */
	public static final String PROBLEMMM_NAME="PROBLEM";
	public static final String PROBLEMMM_URI="http://www.eclipse.org/gmt/2005/Problem";
	
	/*
	 * PARAMETERS Variables
	 */
	public static final String XMLPARAMETERS_MODEL_NAME="parameters";
	public static final String XMLPARAMETERS_LIBRARY_NAME="XMLPARAMETERS";
	public static final String XMLPARAMETERS_LIBRARY_PATH="/libraries/XMLPARAMETERS.asm";
	protected Map<String, String> xmlparameters = new HashMap<String, String>(); // Libraries
	
	protected ExtendedATLTransform(URL transformation) {
		super(transformation);
	}
	
	protected ExtendedATLTransform(URL transformation, int flags) throws IOException, ATLException {
		super(transformation);
		if((flags & (1 << ExtendedATLTransform.LOG)) != 0) enableLOG();
		if((flags & (1 << ExtendedATLTransform.UML)) != 0) enableUML();
		if((flags & (1 << ExtendedATLTransform.MARTE)) != 0)  enableMARTE();
		if((flags & (1 << ExtendedATLTransform.XML)) != 0) enableXML();
		if((flags & (1 << ExtendedATLTransform.PROBLEM)) != 0) enablePROBLEM();
		if((flags & (1 << ExtendedATLTransform.XMLPARAMETERS)) != 0) enableXMLParameters();
	}

	/*
	 * LOG Support
	 */
	public boolean is_enabled_LOG() {
		return libs.containsKey(LOG_LIBRARY_NAME);
	}
	public void enableLOG() {
		Log.verboseMessage(ATLTransform.PluginID, "Enabling LOG facilities");
		// Old way to display messages using ATL LOG library
		if(!libs.containsKey(LOG_LIBRARY_NAME)) this.addLibrary(LOG_LIBRARY_NAME,  ExtendedATLTransform.class.getResource(LOG_LIBRARY_PATH));
		// New way to display messages using com.thalesgroup.com.java.log
		addCustomOperationToVM(LOGATLFacilities.class, "debugMessage");
		addCustomOperationToVM(LOGATLFacilities.class, "verboseMessage");
		addCustomOperationToVM(LOGATLFacilities.class, "demoMessage");
		addCustomOperationToVM(LOGATLFacilities.class, "warningMessage");
		addCustomOperationToVM(LOGATLFacilities.class, "errorMessage");
	}
	
	/*
	 * UML2 Support
	 */
	public boolean is_enabled_UML() {
		return models.containsKey(UML2MM_NAME) && libs.containsKey(UML2_LIBRARY_NAME);
	}
	
	public void enableUML() throws IOException {
		if(!models.containsKey(UML2MM_NAME)) addMetamodelByURI(UML2MM_NAME, UML2MM_URI);
		if(!libs.containsKey(UML2_LIBRARY_NAME)) this.addLibrary(UML2_LIBRARY_NAME,  ExtendedATLTransform.class.getResource(UML2_LIBRARY_PATH));	
	}

	protected void addUMLInputModel(String name, String file)
	throws IOException, NotLoadedException {
		if(!is_enabled_UML()) throw new NotLoadedException("UML");
		this.addInputModel(name, UML2MM_NAME, file);
	}
	protected void addUMLInputModel(String name, IFile file)
	throws IOException, CoreException, NotLoadedException {
		if(!is_enabled_UML()) throw new NotLoadedException("UML");
		this.addInputModel(name, UML2MM_NAME, file);
	}
	protected void addUMLInputModel(String name, URL file)
	throws IOException, NotLoadedException {
		if(!is_enabled_UML()) throw new NotLoadedException("UML");
		this.addInputModel(name, UML2MM_NAME, file);
	}
	protected void addUMLInputModel(String name, InputStream inputstream)
	throws IOException, NotLoadedException {
		if(!is_enabled_UML()) throw new NotLoadedException("UML");
		this.addInputModel(name, UML2MM_NAME, inputstream);
	}
	protected void createUMLModel(String name) throws NotLoadedException {
		if(!is_enabled_UML()) throw new NotLoadedException("UML");
		this.createModel(name, UML2MM_NAME);
	}
	
	/*
	 * MARTE Support
	 */
	public boolean is_enabled_MARTE() throws IOException {
		return is_enabled_UML() && libs.containsKey(MARTE_LIBRARY_NAME);
	}
	public void enableMARTE() throws IOException, ATLException {
		if (marteImplementation == null) {
			String message = "The MARTE implementation has not been precised";
			throw new ATLException(message);
		}
		else {
			enableMARTE(marteImplementation);
		}
	}
	public void enableMARTE(MARTEImplementation marteImplementation) throws IOException, ATLException {
		Log.verboseMessage(ATLTransform.PluginID, "Enabling MARTE facilities");
		enableUML();
		MARTEFacilities.init(marteImplementation);
		if(!libs.containsKey(MARTE_LIBRARY_NAME)) marteImplementation.Load();
		
		if(!libs.containsKey(MARTE_LIBRARY_NAME)) this.addLibrary(MARTE_LIBRARY_NAME,ExtendedATLTransform.class.getResource(marteImplementation.getLibraryPath()));
		
		// Specific operation for getting information from VSLExpression are added to ATL VM
		Log.verboseMessage(ATLTransform.PluginID, "Adding operation for manipulating VSL expressions to ATL VM");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLString");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLString");
		
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLBoolean");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLBoolean");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLBooleanValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLBooleanValue");
		
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLInteger");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLInteger");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLIntegerValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLIntegerValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteCanConvertVSLIntegerValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteConvertVSLIntegerValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasMaximumVSLIntegerValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetMaximumVSLIntegerValue");
		
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLDouble");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLDouble");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasVSLDoubleValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetVSLDoubleValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteCanConvertVSLDoubleValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteConvertVSLDoubleValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteHasMaximumVSLDoubleValue");
		addCustomOperationToVM(MARTEATLFacilities.class, "marteGetMaximumVSLDoubleValue");
	}

	/*
	 * XML IMPORT / EXPORT
	 */
	public boolean is_enabled_XML() {
		return models.containsKey(XMLMM_NAME);
	}
	
	public void enableXML() throws IOException {
		if(getResourceModel(XMLMM_NAME) == null) {
			URL XMLMM_URL = FileLocator.find(Platform.getBundle(XMLMM_BUNDLE),
					new Path(XMLMM_PATH), Collections.EMPTY_MAP);
			this.addMetamodel(XMLMM_NAME, XMLMM_URL);
		} else {
			if(!models.containsKey(XMLMM_NAME)) {
				addMetamodelByURI(XMLMM_NAME, XMLMM_NAME);
			}
		}
	}
	protected String exportModelAsXMLToString(String name) throws IOException, NotLoadedException {
		
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		initEMF();

		Map parameters = Collections.EMPTY_MAP;

		XMLExtractor xmle = new XMLExtractor();
		OutputStream out = new ByteArrayOutputStream();
		ASMModel m = models.get(name);
		xmle.extract(m, out, parameters); // Extract
		out.flush();
		out.close();

		return out.toString();
	}

	protected void exportModelAsXML(String name, IFile file) throws CoreException,
			IOException, NotLoadedException {
		
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		exportStringToIFile(exportModelAsXMLToString(name), file);
	}

	protected void exportModelAsXML(String name, String file) throws IOException, NotLoadedException {
		
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		initEMF();

		Map parameters = Collections.EMPTY_MAP;
		XMLExtractor xmle = new XMLExtractor();
		OutputStream out = new FileOutputStream(file);
		xmle.extract(models.get(name), out, parameters);
		out.flush();
		out.close();
	}

	protected void exportModelAsXML(String name, OutputStream outputstream) throws NotLoadedException {
		
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		initEMF();
		Map parameters = Collections.EMPTY_MAP;
		XMLExtractor xmle = new XMLExtractor();
		xmle.extract(models.get(name), outputstream, parameters); // Extract
	}
	
	protected void importXMLAsModelFromString(String name, String string) throws IOException, NotLoadedException {
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		importXMLAsModel(name, new ByteArrayInputStream(string.getBytes()));
	}
	protected void importXMLAsModel(String name, IFile file) throws CoreException,
			IOException, NotLoadedException {
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		importXMLAsModel(name, file.getContents());
	}
	protected void importXMLAsModel(String name, String file) throws IOException, NotLoadedException {
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		importXMLAsModel(name, new FileInputStream(file));

	}
	protected void importXMLAsModel(String name, InputStream inputstream) throws IOException, NotLoadedException {
		
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		initEMF();
		Map parameters = Collections.EMPTY_MAP;
		XMLInjector xmli = new XMLInjector();
		createXMLModel(name);
		xmli.inject(models.get(name), inputstream, parameters);
	}
	protected void createXMLModel(String name) throws NotLoadedException {
		if(!is_enabled_XML()) throw new NotLoadedException("XML");
		this.createModel(name, XMLMM_NAME);
	}
	
	/*
	 * Problems Support
	 */
	public boolean is_enabled_PROBLEM(){
		return models.containsKey(PROBLEMMM_NAME);
	}
	public void enablePROBLEM() throws IOException {
		addMetamodelByURI(PROBLEMMM_NAME, PROBLEMMM_URI);
	}
	
	protected void createProblemModel(String name) throws NotLoadedException {
		if(!is_enabled_PROBLEM()) throw new NotLoadedException("PROBLEM");
		this.createModel(name, PROBLEMMM_NAME);
	}
	
	/*
	 * XMLPARAMETERS Support
	 */

	public boolean is_enabled_XMLParameters(){
		return is_enabled_XML() && libs.containsKey(XMLPARAMETERS_LIBRARY_NAME);
	}
	public void enableXMLParameters() throws IOException {
		enableXML();
		if(!libs.containsKey(XMLPARAMETERS_LIBRARY_NAME)) this.addLibrary(XMLPARAMETERS_LIBRARY_NAME,ExtendedATLTransform.class.getResource(XMLPARAMETERS_LIBRARY_PATH));
	}
	
	protected void addXMLParameter(String name, String value) {
		xmlparameters.put(name, value);
	}
	
	protected void setXMLParameter(String name, String value) {
		if(hasXMLParameter(name)) {
			removeXMLParameter(name);
		}
		addXMLParameter(name, value);
	}
	
	protected Boolean hasXMLParameter(String name) {
		return xmlparameters.containsKey(name);
	}
	
	protected String removeXMLParameter(String name) {
		return xmlparameters.remove(name);
	}
	
	protected void clearXMLParameters() {
		xmlparameters.clear();
	}
	
	protected void buildXMLParameters() throws NotLoadedException, IOException {
		// Build XML parameters as a string
		String s = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
		s += "<parameters>\n";
		for(String p : xmlparameters.keySet()) {
			s += "  <param name=\""+p+ "\" value=\""+xmlparameters.get(p)+"\"/>\n";
		}
		s += "</parameters>\n";
		
		Log.debugMessage(ATLTransform.PluginID, "Parameters model : "+s);
		
		if(models.containsKey(XMLPARAMETERS_MODEL_NAME)) {
			removeModel(XMLPARAMETERS_MODEL_NAME);
		}
		// Inject the parameters as an input model
		importXMLAsModelFromString(XMLPARAMETERS_MODEL_NAME, s);
		
	}
	
	/*
	 * Launches the transformation
	 */
	@Override
	protected void run() {
		
		if(is_enabled_XMLParameters()) {
			try {
				buildXMLParameters();
			} catch (Exception e) {
				Log.errorMessage(ATLTransform.PluginID, "Error while building XML Parameters", e);
			}
		}
		
		super.run();
		
		if(is_enabled_XMLParameters()) {
			removeModel(XMLPARAMETERS_MODEL_NAME);
		}
	}

	@Override
	protected void run(URL transformation) {
		
		if(is_enabled_XMLParameters()) {
			try {
				buildXMLParameters();
			} catch (Exception e) {
				Log.errorMessage(ATLTransform.PluginID, "Error while building XML Parameters", e);
			}
		}
		
		super.run(transformation);
		
		if(is_enabled_XMLParameters()) {
			removeModel(XMLPARAMETERS_MODEL_NAME);
		}
	}
}
