/**
 * <copyright>
 * Thales ATL (Copyright (c) THALES 2007 All rights reserved)
 * is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published
 * in http://www.eclipse.org/legal/epl-v10.html
 * Thales MARTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 * </copyright>
 *
 * $Id: MARTEATLFacilities.java,v 1.6 2008/06/24 12:34:18 sheywood Exp $
 */
package com.thalesgroup.atl.marte;

/**
 * Methods to use from ATL for MARTE needs
 * @author Eric MAES
 */

import java.util.ArrayList;
import java.util.Iterator;

import org.atl.engine.repositories.emf4atl.ASMEMFModelElement;
import org.atl.engine.vm.StackFrame;
import org.atl.engine.vm.nativelib.ASMBoolean;
import org.atl.engine.vm.nativelib.ASMCollection;
import org.atl.engine.vm.nativelib.ASMInteger;
import org.atl.engine.vm.nativelib.ASMReal;
import org.atl.engine.vm.nativelib.ASMString;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.ValueSpecification;

import com.thalesgroup.atl.ATLFacilities;
import com.thalesgroup.atl.ATLTransform;
import com.thalesgroup.atl.exception.ATLException;
import com.thalesgroup.java.log.Log;

/**
 * Class that is offering facilities for using MARTE in ATL transformations
 * @author maes
 */
public class MARTEATLFacilities extends MARTEFacilities {

	/**
	 * Returns the value (ValueSpecification) of the stereotype attribute of self
	 * @param self
	 * @param stereotypeString
	 * @param propertyString
	 * @return Value (ValueSpecification) of the stereotype attribute of self
	 */
	private static ValueSpecification getValueSpecification(ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString)
		throws ATLException
	{
		ValueSpecification retour = null;
		Element element = ATLFacilities.asmModelElementToElement(self);
		Stereotype stereotype = marteGetStereotype(element, ATLFacilities.asmStringToString(stereotypeString));
		Property property = marteGetProperty(stereotype, ATLFacilities.asmStringToString(propertyString));
		Object value = marteGetPropertyValue(element, stereotype, property);
		if (value instanceof ValueSpecification) {
			retour = (ValueSpecification) value;
		}
		return(retour);
	}
	
	/**
	 * Returns the UnitKind of the stereotype attribute of the element on which the stereotype is applied
	 * @param self Element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the stereotype attribute
	 * @return UnitKind of the stereotype attribute of the element on which the stereotype is applied
	 * @throws ATLException
	 * @deprecated
	 */
	private static Enumeration getUnitKind (ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString)
		throws ATLException
	{
		Enumeration retour = null;
		Element element = ATLFacilities.asmModelElementToElement(self);
		Stereotype stereotype = marteGetStereotype(element, ATLFacilities.asmStringToString(stereotypeString));
		Property property = marteGetProperty(stereotype, ATLFacilities.asmStringToString(propertyString));
		retour = MARTEFacilities.marteGetUnitKind(element, property);
		return retour;
	}
	
	/**
	 * Returns the DataType of the stereotype attribute of the element on which the stereotype is applied
	 * @param self Element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the stereotype attribute
	 * @return DataType of the stereotype attribute of the element on which the stereotype is applied
	 * @throws ATLException
	 */
	private static DataType getDataType (ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString)
		throws ATLException
	{
		DataType retour = null;
		Element element = ATLFacilities.asmModelElementToElement(self);
		Stereotype stereotype = marteGetStereotype(element, ATLFacilities.asmStringToString(stereotypeString));
		Property property = marteGetProperty(stereotype, ATLFacilities.asmStringToString(propertyString));
		retour = MARTEFacilities.marteGetDataType(element, property);
		return retour;
	}
	
	
	/*
	 * Double
	 */

	/**
	 * Returns true if the value (double) of the stereotype attribute of the element can be converted in the desired Unit.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @param finalUnit Unit in which to convert the value
	 * @return If the value (double) of the stereotype attribute of the element can be converted in the desired Unit.
	 */
	public static ASMBoolean marteCanConvertVSLDoubleValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLDoubleValue");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			String unit = ATLFacilities.asmStringToString(finalUnit);
			DataType dataType = getDataType(self, stereotypeString, propertyString);
			retour = new ASMBoolean(MARTEFacilities.marteCanConvertVSLDoubleValue(valueSpecification,dataType,unit));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLDouble", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (double) of the stereotype attribute of the element in the desired Unit.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @param finalUnit Unit in which to convert the value
	 * @return Value (double) of the stereotype attribute of the element in the desired Unit
	 */
	public static ASMReal marteConvertVSLDoubleValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMReal retour = new ASMReal(0);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLDoubleValue");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			String unit = ATLFacilities.asmStringToString(finalUnit);
			DataType dataType = getDataType(self, stereotypeString, propertyString);
			retour = new ASMReal(marteConvertVSLDoubleValue(valueSpecification,dataType,unit));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLDoubleValue", exception);
		}
		return(retour);
	}

	public static ASMBoolean marteHasVSLDoubleValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		return marteCanConvertVSLDoubleValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns the value (double) of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @return Value (double) of the stereotype attribute of the element
	 */
	public static ASMReal marteGetVSLDoubleValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		return marteConvertVSLDoubleValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}

	public static ASMBoolean marteCanConvertMaximumVSLDoubleValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMBoolean retour = new ASMBoolean(true);
		boolean treated = false;
		try {
			check();
			Iterator selfIterator = self.iterator();
			while (selfIterator.hasNext()) {
				Object elementObject = selfIterator.next();
				if (elementObject instanceof ASMEMFModelElement) {
					ASMEMFModelElement asmElement = (ASMEMFModelElement) elementObject;
					if (!treated) {
						treated=true;
						retour = marteCanConvertVSLDoubleValue(frame, asmElement, stereotypeString, propertyString, finalUnit);
					}
					else {
						retour = new ASMBoolean(retour.getSymbol() && marteCanConvertVSLDoubleValue(frame, asmElement, stereotypeString, propertyString, finalUnit).getSymbol());
					}
				}
			}
			if (!treated) {
				String message = "Error determining the possibility to get the maximum value";
				throw new ATLException(message);
			}
		}
		catch(Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteCanConvertMaximumVSLDoubleValue", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the maximum value (double) of the stereotype attribute of the element in the desired Unit.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self List of Model elements
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @param finalUnit Unit in which to convert the value
	 * @return Value (double) of the stereotype attribute of the element in the desired Unit
	 */
	public static ASMReal marteConvertMaximumVSLDoubleValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMReal retour = new ASMReal(0);
		double max = 0;
		boolean treated = false;
		try {
			check();
			Iterator selfIterator = self.iterator();
			while (selfIterator.hasNext()) {
				Object elementObject = selfIterator.next();
				if (elementObject instanceof ASMEMFModelElement) {
					ASMEMFModelElement asmElement = (ASMEMFModelElement) elementObject;
					if (!treated) {
						treated=true;
						max = marteConvertVSLDoubleValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble();
					}
					else {
						if (marteConvertVSLDoubleValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble()>max) {
							max = marteConvertVSLDoubleValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble();
						}
					}
				}
			}
			if (!treated) {
				String message = "Error determining the maximum value";
				throw new ATLException(message);
			}
			retour = new ASMReal(max);
		}
		catch(Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteConvertMaximumVSLDoubleValue", exception);
		}
		return(retour);
	}

	public static ASMBoolean marteHasMaximumVSLDoubleValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString) {
		return marteCanConvertMaximumVSLDoubleValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns the maximum value (double) of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self List of Model elements
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @return Value (double) of the stereotype attribute of the element
	 */
	public static ASMReal marteGetMaximumVSLDoubleValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString) {
		return marteConvertMaximumVSLDoubleValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns true if one of the parameter in attribList of the stereotype attribute of the element has a value as a double.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return If the parameter of the stereotype attribute of the element has a value as a double
	 */	
	public static ASMBoolean marteHasVSLDouble(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteHasVSLDouble");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			if (valueSpecification != null) {
				ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
				retour = new ASMBoolean(marteHasDoubleValue(valueSpecification, attribList));
			}
			else {
				retour = new ASMBoolean(false);
			}
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLDouble", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (double) of one of the parameter in attribList of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribString Name of the parameter to find
	 * @return
	 */
	public static ASMReal marteGetVSLDouble(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMReal retour = new ASMReal(0);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLDouble");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
			retour = new ASMReal(marteGetDoubleValue(valueSpecification, attribList));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLDouble", exception);
		}
		return(retour);
	}

	/*
	 * Integer
	 */

	public static ASMBoolean marteCanConvertVSLIntegerValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLDoubleValue");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			String unit = ATLFacilities.asmStringToString(finalUnit);
			Enumeration unitKind = getUnitKind(self, stereotypeString, propertyString);
			retour = new ASMBoolean(MARTEFacilities.marteCanConvertVSLIntegerValue(valueSpecification,unitKind,unit));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLDouble", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (int) of the stereotype attribute of the element in the desired Unit.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @param finalUnit Unit in which to convert the value
	 * @return Value (int) of the stereotype attribute of the element in the desired Unit
	 */
	public static ASMInteger marteConvertVSLIntegerValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMInteger retour = new ASMInteger(0);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLIntegerValue");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			String unit = ATLFacilities.asmStringToString(finalUnit);
			DataType dataType = getDataType(self, stereotypeString, propertyString);
			retour = new ASMInteger(marteConvertVSLIntegerValue(valueSpecification,dataType,unit));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLDoubleValue", exception);
		}
		return(retour);
	}

	public static ASMBoolean marteHasVSLIntegerValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		return marteCanConvertVSLIntegerValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns the value (int) of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @return Value (int) of the stereotype attribute of the element
	 */
	public static ASMInteger marteGetVSLIntegerValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		return marteConvertVSLIntegerValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}

	public static ASMBoolean marteCanConvertMaximumVSLIntegerValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMBoolean retour = new ASMBoolean(true);
		boolean treated = false;
		try {
			check();
			Iterator selfIterator = self.iterator();
			while (selfIterator.hasNext()) {
				Object elementObject = selfIterator.next();
				if (elementObject instanceof ASMEMFModelElement) {
					ASMEMFModelElement asmElement = (ASMEMFModelElement) elementObject;
					if (!treated) {
						treated=true;
						retour = marteCanConvertVSLIntegerValue(frame, asmElement, stereotypeString, propertyString, finalUnit);
					}
					else {
						retour = new ASMBoolean(retour.getSymbol() && marteCanConvertVSLIntegerValue(frame, asmElement, stereotypeString, propertyString, finalUnit).getSymbol());
					}
				}
			}
			if (!treated) {
				String message = "Error determining the possibility to get the maximum value";
				throw new ATLException(message);
			}
		}
		catch(Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteCanConvertMaximumVSLIntegerValue", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the maximum value (integer) of the stereotype attribute of the element in the desired Unit.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self List of Model elements
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @param finalUnit Unit in which to convert the value
	 * @return Value (integer) of the stereotype attribute of the element in the desired Unit
	 */
	public static ASMInteger marteConvertMaximumVSLIntegerValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString, ASMString finalUnit) {
		ASMInteger retour = new ASMInteger(0);
		int max = 0;
		boolean treated = false;
		try {
			check();
			Iterator selfIterator = self.iterator();
			while (selfIterator.hasNext()) {
				Object elementObject = selfIterator.next();
				if (elementObject instanceof ASMEMFModelElement) {
					ASMEMFModelElement asmElement = (ASMEMFModelElement) elementObject;
					if (!treated) {
						treated=true;
						max = new Double(marteConvertVSLIntegerValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble()).intValue();
					}
					else {
						if (marteConvertVSLIntegerValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble()>max) {
							max = new Double(marteConvertVSLIntegerValue(frame, asmElement, stereotypeString, propertyString, finalUnit).asDouble()).intValue();
						}
					}
				}
			}
			if (!treated) {
				String message = "Error determining the maximum value";
				throw new ATLException(message);
			}
			retour = new ASMInteger(max);
		}
		catch(Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetMaximumVSLDoubleValue", exception);
		}
		return(retour);
	}

	public static ASMBoolean marteHasMaximumVSLIntegerValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString) {
		return marteCanConvertMaximumVSLIntegerValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns the maximum value (integer) of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self List of Model elements
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @return Value (integer) of the stereotype attribute of the element
	 */
	public static ASMInteger marteGetMaximumVSLIntegerValue(StackFrame frame, ASMCollection self, ASMString stereotypeString, ASMString propertyString) {
		return marteConvertMaximumVSLIntegerValue(frame, self, stereotypeString, propertyString, new ASMString(LITERAL_DEFAULT));
	}
	
	/**
	 * Returns true if one of the parameter in attribList of the stereotype attribute of the element has a value as an int.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return If the parameter of the stereotype attribute of the element has a value as an int
	 */
	public static ASMBoolean marteHasVSLInteger(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteHasVSLInteger("+stereotypeString.toString()+","+propertyString+","+attribStringList.toString()+")");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			if (valueSpecification != null) {
				ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
				retour = new ASMBoolean(marteHasIntegerValue(valueSpecification, attribList));
			} else {
				retour = new ASMBoolean(false);
			}
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLInteger for "+self.toString()+","+stereotypeString.toString()+","+propertyString+","+attribStringList.toString(), exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (int) of one of the parameter in attribList of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return Value (int) of the parameter of the stereotype attribute of the element
	 */
	public static ASMInteger marteGetVSLInteger(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMInteger retour = new ASMInteger(0);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLInteger("+stereotypeString.toString()+","+propertyString+","+attribStringList.toString()+")");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
			retour = new ASMInteger(marteGetIntegerValue(valueSpecification, attribList));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLInteger for "+self.toString()+","+stereotypeString.toString()+","+propertyString+","+attribStringList.toString(), exception);
		}
		return(retour);
	}

	/*
	 * Boolean
	 */

	/**
	 * @deprecated
	 */
	public static ASMBoolean marteHasVSLBooleanValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLBooleanValue");
			Element element = ATLFacilities.asmModelElementToElement(self);
			Stereotype stereotype = marteGetStereotype(element, ATLFacilities.asmStringToString(stereotypeString));
			Property property = marteGetProperty(stereotype, ATLFacilities.asmStringToString(propertyString));
		
			ValueSpecification valueValueSpecification = marteGetVSLValue(element, stereotype, property);
			boolean value = marteHasBooleanValue(valueValueSpecification, null);
			Log.debugMessage(ATLTransform.PluginID, "Value of "+propertyString+" : "+value);
		
			retour = new ASMBoolean(value);
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLBooleanValue", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (boolean) of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param stereotypeString Name of the Stereotype
	 * @param propertyString Name of the Stereotype Attribute
	 * @return Value (boolean) of the stereotype attribute of the element
	 */
	public static ASMBoolean marteGetVSLBooleanValue(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLBooleanValue");
			DataType nfpType = getDataType(self, stereotypeString, propertyString);
			Element element = ATLFacilities.asmModelElementToElement(self);
			ArrayList<String> attribList = new ArrayList<String>();
			attribList.add(marteGetValueAttrib(nfpType));
			ValueSpecification valueValueSpecification = marteGetVSL(element, attribList);
			boolean value = marteGetBooleanValue(valueValueSpecification, null);
			Log.debugMessage(ATLTransform.PluginID, "Value of "+propertyString+" : "+value);
		
			retour = new ASMBoolean(value);
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLBooleanValue", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns true if one of the parameter in attribList of the stereotype attribute of the element has a value as a boolean.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return If the parameter of the stereotype attribute of the element has a value as a boolean
	 */
	public static ASMBoolean marteHasVSLBoolean(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteHasVSLBoolean");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			if (valueSpecification != null) {
				ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
				retour = new ASMBoolean(marteHasBooleanValue(valueSpecification, attribList));
			} else {
				retour = new ASMBoolean(false);
			}
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLBoolean", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (boolean) of one of the parameter in attribList of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return Value (boolean) of the parameter of the stereotype attribute of the element
	 */
	public static ASMBoolean marteGetVSLBoolean(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLBoolean");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
			retour = new ASMBoolean(marteGetBooleanValue(valueSpecification, attribList));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLInteger", exception);
		}
		return(retour);
	}

	/*
	 * String
	 */

	/**
	 * Returns true if one of the parameter in attribList of the stereotype attribute of the element has a value as a String.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return If the parameter of the stereotype attribute of the element has a value as a Strin
	 */
	public static ASMBoolean marteHasVSLString(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMBoolean retour = new ASMBoolean(false);
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteHasVSLString");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			if (valueSpecification != null) {
				ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
				retour = new ASMBoolean(marteHasStringValue(valueSpecification, attribList));
			} else {
				retour = new ASMBoolean(false);
			}
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteHasVSLString", exception);
		}
		return(retour);
	}
	
	/**
	 * Returns the value (String) of one of the parameter in attribList of the stereotype attribute of the element.
	 * @param frame Necessary parameter for a use from ATL
	 * @param self Model element
	 * @param attribStringList List of names of the attribute to find
	 * @return Value (String) of the parameter of the stereotype attribute of the element
	 */
	public static ASMString marteGetVSLString(StackFrame frame, ASMEMFModelElement self, ASMString stereotypeString, ASMString propertyString, ASMCollection attribStringList) {
		ASMString retour = new ASMString("");
		try {
			check();
			Log.verboseMessage(ATLTransform.PluginID, "marteGetVSLString");
			ValueSpecification valueSpecification = getValueSpecification(self,stereotypeString,propertyString);
			ArrayList<String> attribList = ATLFacilities.asmStringCollectionToStringList(attribStringList);
			retour = new ASMString(marteGetStringValue(valueSpecification, attribList));
		}
		catch (Exception exception) {
			Log.errorMessage(ATLTransform.PluginID, "Error in marteGetVSLInteger", exception);
		}
		return(retour);
	}
	
}
