/**
 * <copyright>
 * Thales ATL (Copyright (c) THALES 2007 All rights reserved)
 * is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published
 * in http://www.eclipse.org/legal/epl-v10.html
 * Thales MARTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 * </copyright>
 *
 * $Id: MARTEFacilities.java,v 1.9 2008/06/24 12:34:18 sheywood Exp $
 */
package com.thalesgroup.atl.marte;
/**
 * Methods to use for MARTE needs
 * @author Eric MAES
 */

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralNull;
import org.eclipse.uml2.uml.LiteralSpecification;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;

import VSL.ChoiceSpecification;
import VSL.CollectionSpecification;
import VSL.ConditionalExpression;
import VSL.EnumerationSpecification;
import VSL.IntervalSpecification;
import VSL.LiteralDateTime;
import VSL.LiteralDefault;
import VSL.LiteralReal;
import VSL.TimeExpression;
import VSL.TupleItemValue;
import VSL.TupleSpecification;
import VSL.Variable;

import com.cea.nfp.parsers.modelgenerator.IModelFacade;
import com.cea.nfp.parsers.modelgenerator.VSLFullParser;
import com.thalesgroup.atl.ATLTransform;
import com.thalesgroup.atl.exception.ATLException;
import com.thalesgroup.java.log.Log;
import com.thalesgroup.marte.vsl.VSLDate;

/**
 * Class that is offering facilities for using MARTE 
 * @author maes
 */
public abstract class MARTEFacilities {
	
	/**
	 * MARTE default constant value
	 */
	public final static String LITERAL_DEFAULT = "DEFAULT";	
	/**
	 * MARTE null constant value
	 */
	public final static String LITERAL_NULL = "NULL";
	
	/**
	 * MARTE implementation
	 */
	protected static MARTEImplementation marte;
	
	/**
	 * Initialization
	 * @param marteImplementation Kind of MARTE implementation
	 */
	public static void init(MARTEImplementation marteImplementation) throws ATLException {
		marte=marteImplementation;
	}
	
	/**
	 * Checks that an interface has been initialized for MARTE
	 * init has to be called before any call to the marte MARTEInterface or marteImplementation
	 * @return true
	 * @throws ATLException
	 */
	protected static boolean check() throws ATLException {
		Log.debugMessage(ATLTransform.PluginID, marte+" checking...");
		if (marte==null) {
			String message = "The MARTE interface has not been initialized";
			throw new ATLException(message);
		}
		return true;
	}

	/**
	 * Return true if the stereotype is applied on the element
	 * @param element Element on which to test if the stereotype is applied
	 * @param stereotypeString Name of the stereotype
	 * @return If the stereotype is applied on the element
	 * @throws ATLException
	 */
	public static boolean marteIsTypeOf(Element element, String stereotypeString)
		throws ATLException
	{
		if (element.getAppliedStereotype(stereotypeString) != null) {
			return true;
		}
		return false;
	}
	public static boolean marteIsTypeOf(Element element, Stereotype stereotype)
		throws ATLException
	{
		return marteIsTypeOf(element,stereotype.getQualifiedName());
	}

	/**
	 * Return the stereotype which name is stereotypeString and which is applied on element
	 * @param element UML Element on which the stereotype is applied
	 * @param stereotypeString Name of the stereotype
	 * @return The stereotype applied to the element
	 * @throws ATLException
	 */
	public static Stereotype marteGetStereotype(Element element, String stereotypeString) throws ATLException {
		Stereotype retour = element.getAppliedStereotype(stereotypeString);
		if (retour==null) {
			String message = "Stereotype not found : "+stereotypeString+" for element "+element;
			throw new ATLException(message);
		}
		return (retour);
	}

	/**
	 * Return the stereotype attribute which name is propertyString
	 * @param stereotype The stereotype that owns the attribute
	 * @param propertyString Name of the stereotype attribute
	 * @return The stereotype attribute which name is propertyString
	 * @throws ATLException
	 */	
	public static Property marteGetProperty(Stereotype stereotype, String propertyString) 
		throws ATLException
	{
		Property retour = null;
		Iterator properties = stereotype.getAllAttributes().iterator();
		while (properties.hasNext()) {
			Object propertyObject = properties.next();
			if (propertyObject instanceof Property) {
				Property property = (Property) propertyObject;
				if (property.getName().equals(propertyString)) {
					retour = property;
				}
			}
		}
		if (retour==null) {
			String message = "Attribute stereotype not found : "+propertyString+"for stereotype "+stereotype.getName();
			throw new ATLException(message);
		}
		return (retour);
	}
	
	/**
	 * Return the value of an element stereotype attribute
	 * @param element Element on which the stereotype is applied
	 * @param stereotype Stereotype that is applied
	 * @param property Stereotype attribute
	 * @return Value of element stereotype attribute
	 */
	public static Object marteGetPropertyValue(Element element, Stereotype stereotype, Property property)
		throws ATLException
	{
		Object retour = null;
		if (marteIsTypeOf(element, stereotype)) {
			Object propertyValueObject = element.getValue(stereotype, property.getName());
			if (propertyValueObject instanceof Property) {
				retour = propertyValueObject;
			}
			if (retour==null) {
				retour = marte.getPropertyValue(element, property, propertyValueObject);
			}
		}
		return(retour);
	}
	
	/**
	 * Return the value of an element stereotype attribute
	 * @param element Element on which the stereotype is applied
	 * @param stereotypeString Name of the stereotype that is applied
	 * @param propertyString Name of the stereotype attribute
	 * @return Value of element stereotype attribute
	 */	
	public static Object marteGetPropertyValue(Element element, String stereotypeString, String propertyString)
		throws ATLException
	{
		Stereotype stereotype = marteGetStereotype(element, stereotypeString);
		Property property = marteGetProperty(stereotype, propertyString);
		return marteGetPropertyValue(element, stereotype, property);
	}
	
	/**
	 * Return the VSL expression of the property of the stereotype applied on element
	 * @param element UML element on which is applied the stereotype
	 * @param stereotype 
	 * @param property Strreotype attribute
	 * @return VSL expression
	 * @throws ATLException
	 * @deprecated
	 */
	public static String marteGetPropertyVSLValue(Element element, Stereotype stereotype, Property property)
		throws ATLException
	{
		String retour = null;
		Object vslObject = marteGetPropertyValue(element,stereotype,property);
		if (vslObject instanceof String) {
			retour = (String) vslObject;
		}
		if (retour==null) {
			String message = "Error in getting a VSL value";
			throw new ATLException(message); 
		}
		return(retour);
	}
	
	/**
	 *
	 * @param element
	 * @param stereotypeString
	 * @param propertyString
	 * @return
	 * @throws ATLException
	 * @deprecated
	 */
	public static String marteGetPropertyVSLValue(Element element, String stereotypeString, String propertyString)
		throws ATLException
	{
		Stereotype stereotype = marteGetStereotype(element, stereotypeString);
		Property property = marteGetProperty(stereotype, propertyString);
		return marteGetPropertyVSLValue(element, stereotype, property);
	}
	
	/**
	 * Returns the DataType of the stereotype attribute
	 * @param element Element which has the stereotype attribute
	 * @param property Stereotype attribute
	 * @return DataType of the stereotype attribute
	 */
	public static DataType marteGetDataType(Element element, Property property)
		throws ATLException
	{
		DataType retour = null;
		IModelFacade facade = marte.getFacade(element);
		DataType propertyType = facade.typeof(property);
		retour = propertyType;
		return retour;
	}
	
	/**
	 * Returns the ValueSpecification of the attribute of the stereotype applied to element
	 * @param element Element on which is applied the stereotype
	 * @param stereotype Stereotype that is applied
	 * @param property Stereotype attribute
	 * @param vslExpression String representing the ValueSpecification
	 * @return ValueSpecification of the attribute of the stereotype applied to element
	 */
	public static ValueSpecification marteGetValueSpecification(Element element, Property property, String vslExpression)
		throws ATLException
	{
		ValueSpecification retour = null;
		IModelFacade facade = marte.getFacade(element);
		DataType propertyType = marteGetDataType(element,property);
		try {
			retour = VSLFullParser.parse(vslExpression,propertyType,facade);
		}
		catch(Exception exception) {
			throw new ATLException("Error in parsing vslExpression "+vslExpression);
		}
		return retour;
	}
	
	/**
	 * Returns the value of the valueSpecification corresponding to an attribute in the attribute list.
	 * @param valueSpecification ValueSpecification to get the attribute value from
	 * @param attribList List of attribute names corresponding to fields defined for TupleTypes, CollectionsTypes and ChoiceTypes
	 * @return Value of the valueSpecification corresponding to an attribute in the attribute list
	 * @throws ATLException
	 */
	public static ValueSpecification marteGetValueSpecification(ValueSpecification valueSpecification, ArrayList<String> attribList)
		throws ATLException
	{
		ValueSpecification retour = null;
		if (valueSpecification instanceof TupleSpecification) {
			TupleSpecification tupleSpecification = (TupleSpecification) valueSpecification;
			Iterator tupleItemValues = tupleSpecification.getTupleItem().iterator();
			while (tupleItemValues.hasNext()) {
				Object tupleItemValueObject = tupleItemValues.next();
			
				if (tupleItemValueObject instanceof TupleItemValue) {
					TupleItemValue tupleItemValue = (TupleItemValue) tupleItemValueObject;
					Log.debugMessage(ATLTransform.PluginID, "TupleItem Name : "+tupleItemValue.getTupleItemName());
					if (attribList.contains(tupleItemValue.getTupleItemName())) {
						ValueSpecification itemValue = tupleItemValue.getItemValue();
						retour = itemValue;
					}
				}
				else {
					String message = "Element is not a TupleItemValue";
					throw new ATLException(message);
				}
			}
		}
		if (valueSpecification instanceof CollectionSpecification) {
			CollectionSpecification collectionSpecification = (CollectionSpecification) valueSpecification;
			Iterator collectionItemValues = collectionSpecification.getItemValue().iterator();
			while (collectionItemValues.hasNext()) {
				Object collectionItemValueObject = collectionItemValues.next();
			
				if (collectionItemValueObject instanceof ValueSpecification) {
					ValueSpecification collectionItemValue = (ValueSpecification) collectionItemValueObject;
					if (marteGetValueSpecification(collectionItemValue,attribList)!=null) {
						retour = marteGetValueSpecification(collectionItemValue,attribList);
					}
				}
				else {
					String message = "Element is not a ValueSpecification";
					throw new ATLException(message);
				}
			}
		}
		if (valueSpecification instanceof ChoiceSpecification) {
			ChoiceSpecification choiceSpecification = (ChoiceSpecification) valueSpecification;
			if (attribList.contains(choiceSpecification.getChosenAlternative())) {
				retour = choiceSpecification.getValue();
			}
		}
		return retour;
	}
	
	/**
	 * Returns the ValueSpecification corresponding to the expr of the property of the stereotype applied to element
	 * The property type is supposed to be stereotyped by <<NfpType>>. 
	 * @param element
	 * @param stereotype
	 * @param property
	 * @return
	 * @throws ATLException
	 * @deprecated
	 */
	public static ValueSpecification marteGetVSLExpr(Element element, Stereotype stereotype, Property property)
		throws ATLException
	{
		ValueSpecification retour = null;
		
		DataType propertyType = marteGetDataType(element, property);
		Object exprAttribObject = marteGetPropertyValue(propertyType,marte.getConstants().NfpType,marte.getConstants().NfpType_exprAttrib);
		if (exprAttribObject instanceof NamedElement) {
			String exprAttrib = ((NamedElement)exprAttribObject).getName();
			String vslExpression = marteGetPropertyVSLValue(element, stereotype, property);
			ValueSpecification propertyValueSpecification = marteGetValueSpecification(element,property,vslExpression);
			ArrayList<String> attribList = new ArrayList<String>();
			attribList.add(exprAttrib);
			retour = marteGetValueSpecification(propertyValueSpecification, attribList);
		}

		if (retour==null) {
			String message = "Information on expr has not been found for "+element;
			throw new ATLException(message);
		}
		
		return retour;
	}
	
	/**
	 * Returns the value of the attribute of nfpType
	 * @param nfpType The Nfp_Type to get the attribute value from
	 * @param attrib The attribute name to get the value
	 * @return Value of the attribute of nfpType
	 * @throws ATLException
	 */
	protected static String marteGetAttrib(DataType nfpType, String attrib)
		throws ATLException
	{
		String retour = null;
		Object attribObject = marteGetPropertyValue(nfpType,marte.getConstants().NfpType,attrib);
		if (attribObject instanceof NamedElement) {
			retour = ((NamedElement) attribObject).getName();
		}
		return (retour);
	}
	
	/**
	 * Returns the value of the exprAttrib of nfpType
	 * @param nfpType The Nfp_Type to get the attribute value from
	 * @return Value of the exprAttrib of nfpType
	 * @throws ATLException
	 */
	public static String marteGetExprAttrib (DataType nfpType)
		throws ATLException
	{	
		return(marteGetAttrib(nfpType, marte.getConstants().NfpType_exprAttrib));
	}
	
	/**
	 * Returns the value of the valueAttrib of nfpType
	 * @param nfpType The Nfp_Type to get the attribute value from
	 * @return Value of the valueAttrib of nfpType
	 * @throws ATLException
	 */
	public static String marteGetValueAttrib (DataType nfpType)
		throws ATLException
	{	
		return(marteGetAttrib(nfpType, marte.getConstants().NfpType_valueAttrib));
	}
	
	/**
	 * Returns the value of the unitAttrib of nfpType
	 * @param nfpType The Nfp_Type to get the attribute value from
	 * @return Value of the unitAttrib of nfpType
	 * @throws ATLException
	 */
	public static String marteGetUnitAttrib (DataType nfpType)
		throws ATLException
	{	
		return(marteGetAttrib(nfpType, marte.getConstants().NfpType_unitAttrib));
	}

	/**
	 * Returns the ValueSpecification corresponding to the value of the property of the stereotype applied to element
	 * The property type is supposed to be stereotyped by <<NfpType>>. 
	 * @param element
	 * @param stereotype
	 * @param property
	 * @return
	 * @throws ATLException
	 * @deprecated
	 */
	public static ValueSpecification marteGetVSLValue(Element element, Stereotype stereotype, Property property)
		throws ATLException
	{
		ValueSpecification retour = null;
		
//		DataType propertyType = marteGetDataType(element, property);
//		Object valueAttribObject = marteGetPropertyValue(propertyType,marte.NfpType(),marte.NfpType_valueAttrib());
//		if (valueAttribObject instanceof NamedElement) {
//			String exprAttrib = ((NamedElement)valueAttribObject).getName();
			//EM ToDo Identified bug in MARTE_RSx 0.0.6
			//EM ToDo Question sent to CEA
			String valueAttrib = "value";
		
			String vslExpression = marteGetPropertyVSLValue(element, stereotype, property);

			ValueSpecification propertyValueSpecification = marteGetValueSpecification(element,property,vslExpression);
			ArrayList<String> attribList = new ArrayList<String>();
			attribList.add(valueAttrib);
			retour = marteGetValueSpecification(propertyValueSpecification, attribList);
//		}

		if (retour==null) {
			String message = "Information on value has not been found for "+element;
			throw new ATLException(message);
		}
		
		return retour;
	}
	
	/**
	 * Returns the ValueSpecification corresponding to the unit of the property of the stereotype applied to element
	 * The property type is supposed to be stereotyped by <<NfpType>>. 
	 * @param element
	 * @param stereotype
	 * @param property
	 * @return
	 * @throws ATLException
	 * @deprecated
	 */
	public static ValueSpecification marteGetVSLUnit(Element element, Stereotype stereotype, Property property)
		throws ATLException
	{
		ValueSpecification retour = null;

		DataType propertyType = marteGetDataType(element, property);
		Object unitAttribObject = marteGetPropertyValue(propertyType,marte.getConstants().NfpType,marte.getConstants().NfpType_unitAttrib);
		if (unitAttribObject instanceof NamedElement) {
			String unitAttrib = ((NamedElement)unitAttribObject).getName();
			String vslExpression = marteGetPropertyVSLValue(element, stereotype, property);
			ValueSpecification propertyValueSpecification = marteGetValueSpecification(element,property,vslExpression);
			ArrayList<String> attribList = new ArrayList<String>();
			attribList.add(unitAttrib);
			retour = marteGetValueSpecification(propertyValueSpecification, attribList);
		}

		if (retour==null) {
			String message = "Information on unit has not been found for "+element;
			throw new ATLException(message);
		}
		
		return retour;
	}

	/**
	 * Returns the ValueSpecification corresponding to one of the attribList attribute of element.
	 * @param element Element to get the ValueSpecification from
	 * @param attribList List of names of attributes
	 * @return ValueSpecification corresponding to one of the attribList attribute of element
	 * @throws ATLException
	 */
	public static ValueSpecification marteGetVSL(Element element, ArrayList<String> attribList)
		throws ATLException
	{
		ValueSpecification retour = null;
		
		if (element instanceof ValueSpecification) {
			ValueSpecification valueSpecification = (ValueSpecification) element;
			retour = marteGetValueSpecification(valueSpecification, attribList);
			
		}
		if (retour==null) {
			String message = "Information on none attributes of "+attribList.toString()+" has been found for "+element;
			throw new ATLException(message);
		}
		
		return retour;
	}
	
	/**
	 * Return the Enumeration corresponding to the definition of a NFP type
	 * @param type
	 * @return
	 * @throws ATLException
	 * @deprecated
	 */
	public static Enumeration marteGetUnitKind(Element element, Property property)
		throws ATLException
	{
		DataType type = marteGetDataType(element, property);		

		Enumeration retour = null;
		Object unitAttribObject = marteGetPropertyValue(type,marte.getConstants().NfpType,marte.getConstants().NfpType_unitAttrib);
		if (unitAttribObject instanceof Property) {
			Property unit = (Property) unitAttribObject;
			Iterator properties = type.getAllAttributes().iterator();
			while(properties.hasNext()) {
				Object propertyObject = properties.next();
				if (propertyObject instanceof Property) {
					Property unitProperty = (Property) propertyObject;
					if (unitProperty.getName().equals(unit.getName())) {
						Type unitKind = unitProperty.getType();
						if (unitKind instanceof Enumeration) {
							retour = (Enumeration) unitKind;
						}
					}
				}
			}
		}
		if (retour==null) {
			String message = "Error in getting the "+type.getName()+" unitAttrib Enumeration";
			new ATLException(message);
		}
		
		return retour;
	}
	
	/**
	 * Return the Enumeration corresponding to the UnitKind of nfpType
	 * @param nfpType NfpType to get the UnitKind from
	 * @return Enumeration corresponding to the UnitKind of nfpType
	 * @throws ATLException
	 */
	public static Enumeration marteGetUnitKind(DataType nfpType)
		throws ATLException
	{
		Enumeration retour = null;
		Object unitAttribObject = marteGetPropertyValue(nfpType,marte.getConstants().NfpType,marte.getConstants().NfpType_unitAttrib);
		if (unitAttribObject instanceof Property) {
			Property unit = (Property) unitAttribObject;
			Iterator properties = nfpType.getAllAttributes().iterator();
			while(properties.hasNext()) {
				Object propertyObject = properties.next();
				if (propertyObject instanceof Property) {
					Property unitProperty = (Property) propertyObject;
					if (unitProperty.getName().equals(unit.getName())) {
						Type unitKind = unitProperty.getType();
						if (unitKind instanceof Enumeration) {
							retour = (Enumeration) unitKind;
						}
					}
				}
			}
		}
		if (retour==null) {
			String message = "Error in getting the "+nfpType.getName()+" unitAttrib Enumeration";
			new ATLException(message);
		}
		
		return retour;
	}
	
	/**
	 * Return the EnumerationLiteral corresponding to a unit
	 * @param enumeration
	 * @param unitString
	 * @return
	 * @throws ATLException
	 */
	public static EnumerationLiteral marteGetUnit(Enumeration enumeration, String unitString)
		throws ATLException
	{
		EnumerationLiteral retour = null;
		Iterator enumerationLiterals = enumeration.getAttributes().iterator();
		while (enumerationLiterals.hasNext()) {
			Object enumerationLiteralObject = enumerationLiterals.next();
			if (enumerationLiteralObject instanceof EnumerationLiteral) {
				EnumerationLiteral enumerationLiteral = (EnumerationLiteral) enumerationLiteralObject;
				if (enumerationLiteral.getName()==unitString) {
					retour = enumerationLiteral;
				}
			}
		}
		if (retour==null) {
			String message = "Error in getting the "+unitString+" EnumerationLiteral of "+enumeration.getName();
			new ATLException(message);
		}
		return retour;
	}

	/*
	 * Boolean
	 */
	
	/**
	 * Returns true if a value (boolean) is corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib Name of the attribute to get if any
	 * @return If a value (boolean) is corresponding to the valueSpecification
	 */
	public static boolean marteHasBooleanValue (ValueSpecification valueSpecification, ArrayList<String> attribList) {
		boolean retour = true;
		try {
			marteGetBooleanValue(valueSpecification, attribList);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Returns a boolean corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib Name of the attribute to get if any
	 * @return Boolean value of valueSpecification
	 * @throws ATLException
	 */
	public static boolean marteGetBooleanValue(ValueSpecification valueSpecification, ArrayList<String> attribList) throws ATLException {
		boolean treated = false;
		boolean retour=false;
		if (valueSpecification instanceof LiteralSpecification) {
			LiteralSpecification literalSpecification = (LiteralSpecification) valueSpecification;
			if (literalSpecification instanceof LiteralString) {
				LiteralString literalString = (LiteralString) literalSpecification;
				treated = true;
				if (literalString.getValue()=="true") {
					retour = true;
				} else {
					retour = false;
				}
			}
			if (literalSpecification instanceof LiteralBoolean) {
				LiteralBoolean literalBoolean = (LiteralBoolean) literalSpecification;
				treated = true;
				retour = literalBoolean.booleanValue();
			}
			if (literalSpecification instanceof LiteralDateTime) {
				LiteralDateTime literalDateTime = (LiteralDateTime) literalSpecification;
				treated = true;
				VSLDate v= literalDateTime.getValue();
				if (v.centisec==0 && v.sec== 0 && v.min==0 && v.hour==0 && v.dayOfMonth==0 && v.month==0 && v.year==0) {
					retour = false;
				} else {
					retour = true;
				}
			}
			if (literalSpecification instanceof LiteralNull) {
				treated = true;
				retour = false;
			}
			if (literalSpecification instanceof LiteralDefault) {
				treated = true;
				retour = false;
			}
			if (literalSpecification instanceof LiteralInteger) {
				int intValue = marteGetIntegerValue(valueSpecification, attribList);
				treated = true;
				if (intValue==0) {
					retour = false;
				} else {
					retour = true;
				}
			}
			if (literalSpecification instanceof LiteralUnlimitedNatural) {
				int intValue = marteGetIntegerValue(valueSpecification, attribList);
				treated = true;
				if (intValue==0) {
					retour = false;
				} else {
					retour = true;
				}
			}
			if (literalSpecification instanceof LiteralReal) {
				int intValue = marteGetIntegerValue(valueSpecification, attribList);
				treated = true;
				if (intValue==0) {
					retour = false;
				} else {
					retour = true;
				}
			}
		}
		if (valueSpecification instanceof EnumerationSpecification) {
			EnumerationSpecification enumerationSpecification = (EnumerationSpecification) valueSpecification;
			return marteGetBooleanValue(enumerationSpecification.getNameExpression(),attribList);
		}
		//TODO Treat OpaqueExpression
//		if (valueSpecification instanceof Expression) {
			//TODO Treat VariableCallExpression
			//TODO Treat Variable
			//TODO Treat PropertyCallExpression
			//TODO Treat OperationCallExpression
			if (valueSpecification instanceof ConditionalExpression) {
				ConditionalExpression conditionalExpression = (ConditionalExpression) valueSpecification;
				boolean condition = marteGetBooleanValue(conditionalExpression.getConditionExpr(), attribList);
				if (condition) {
					treated=true;
					retour = marteGetBooleanValue(conditionalExpression.getIfTrueExpr(), attribList);
				} else {
					treated=true;
					retour = marteGetBooleanValue(conditionalExpression.getIfFalseExpr(), attribList);
				}
			}
//		}
		if (valueSpecification instanceof IntervalSpecification) {
			IntervalSpecification intervalSpecification = (IntervalSpecification) valueSpecification;
			boolean min = marteGetBooleanValue(intervalSpecification.getMin(), attribList);
			boolean max = marteGetBooleanValue(intervalSpecification.getMax(), attribList);
			treated=true;
			retour = min && max;
		}
		if (valueSpecification instanceof TupleSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetBooleanValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof CollectionSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetBooleanValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof ChoiceSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetBooleanValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof TimeExpression) {
			TimeExpression timeExpression = (TimeExpression) valueSpecification;
			if (timeExpression.getExpr()!=null) {
				treated = true;
				retour = marteGetBooleanValue(timeExpression.getExpr(), attribList);
			}
			//TODO treat else case
		}
		if (!treated) {
			String message = "Error in interpreting a value specification as a boolean expression : "+valueSpecification.toString();
			throw new ATLException(message);
		}
		return retour;
	}


	/*
	 * Integer
	 */

	/**
	 * Returns true if a value (int) is corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib Name of the attribute to get if any
	 * @return If a value (int) is corresponding to the valueSpecification
	 */
	public static boolean marteHasIntegerValue (ValueSpecification valueSpecification, ArrayList<String> attribList) {
		boolean retour = true;
		try {
			marteGetIntegerValue(valueSpecification, attribList);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Returns an int corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib NAme of the attribute to get if any
	 * @returna Int value of valueSpecification
	 * @throws ATLException
	 */
	//TODO E.M. Treat Expressions
	public static int marteGetIntegerValue(ValueSpecification valueSpecification, ArrayList<String> attribList) throws ATLException {
		boolean treated = false;
		int retour=0;
		if (valueSpecification instanceof LiteralSpecification) {
			LiteralSpecification literalSpecification = (LiteralSpecification) valueSpecification;
			if (literalSpecification instanceof LiteralString) {
				LiteralString literalString = (LiteralString) literalSpecification;
				treated = true;
				retour = new Integer(literalString.getValue()).intValue();
			}
			if (literalSpecification instanceof LiteralBoolean) {
				LiteralBoolean literalBoolean = (LiteralBoolean) literalSpecification;
				treated = true;
				retour = literalBoolean.integerValue();
			}
			if (literalSpecification instanceof LiteralDateTime) {
				LiteralDateTime literalDateTime = (LiteralDateTime) literalSpecification;
				treated = true;
				VSLDate v= literalDateTime.getValue();
				retour = (((((v.year*12+v.month)*30+v.dayOfMonth)*24+v.hour)*60+v.min)*60+v.sec)*100+v.centisec;
			}
			if (literalSpecification instanceof LiteralNull) {
				treated = true;
				retour = 0;
			}
			if (literalSpecification instanceof LiteralDefault) {
				treated = true;
				retour = 0;
			}
			if (literalSpecification instanceof LiteralInteger) {
				LiteralInteger literalInteger = (LiteralInteger) literalSpecification;
				treated = true;
				retour = literalInteger.getValue();
			}
			if (literalSpecification instanceof LiteralUnlimitedNatural) {
				LiteralUnlimitedNatural literalUnlimitedNatural = (LiteralUnlimitedNatural) literalSpecification;
				treated=true;
				return literalUnlimitedNatural.getValue();
			}
			if (literalSpecification instanceof LiteralReal) {
				LiteralReal literalReal = (LiteralReal) literalSpecification;
				treated=true;
				return new Double(literalReal.getValue()).intValue();
			}
		}
		if (valueSpecification instanceof EnumerationSpecification) {
			EnumerationSpecification enumerationSpecification = (EnumerationSpecification) valueSpecification;
			return marteGetIntegerValue(enumerationSpecification.getNameExpression(), attribList);
		}
		//TODO Treat OpaqueExpression
//		if (valueSpecification instanceof Expression) {
			//TODO Treat VariableCallExpression
			//TODO Treat Variable
			if (valueSpecification instanceof Variable) {
				Variable variable = (Variable) valueSpecification;
				if (variable.getInitExpression() != null) {
					treated = true;
					retour = marteGetIntegerValue(variable.getInitExpression(), attribList);
				} else {
					treated = true;
					retour = -1;
				}
			}
			//TODO Treat PropertyCallExpression
			//TODO Treat OperationCallExpression
			if (valueSpecification instanceof ConditionalExpression) {
				ConditionalExpression conditionalExpression = (ConditionalExpression) valueSpecification;
				boolean condition = marteGetBooleanValue(conditionalExpression.getConditionExpr(), attribList);
				if (condition) {
					treated=true;
					retour = marteGetIntegerValue(conditionalExpression.getIfTrueExpr(), attribList);
				} else {
					treated=true;
					retour = marteGetIntegerValue(conditionalExpression.getIfFalseExpr(), attribList);
				}
			}
//		}
		if (valueSpecification instanceof IntervalSpecification) {
			IntervalSpecification intervalSpecification = (IntervalSpecification) valueSpecification;
			int min = marteGetIntegerValue(intervalSpecification.getMin(), attribList);
			int max = marteGetIntegerValue(intervalSpecification.getMax(), attribList);
			treated = true;
			retour = (max + min)/2;
		}
		if (valueSpecification instanceof TupleSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetIntegerValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof CollectionSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetIntegerValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof ChoiceSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetIntegerValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof TimeExpression) {
			TimeExpression timeExpression = (TimeExpression) valueSpecification;
			if (timeExpression.getExpr()!=null) {
				treated = true;
				retour = marteGetIntegerValue(timeExpression.getExpr(), attribList);
			}
			//TODO treat else case
		}
		if (!treated) {
			String message = "Error in interpreting a value specification as an integer expression";
			throw new ATLException(message);
		}
		return retour;
	}
	
	/**
	 * Converts an integer value in unit to a value in finalUnit
	 * @param enumeration
	 * @param value
	 * @param unit
	 * @param finalUnit
	 * @return Value converted into finalUnit
	 * @throws ATLException
	 */
	public static int marteConvertIntegerValue(Enumeration enumeration, int value, EnumerationLiteral unit, EnumerationLiteral finalUnit)
		throws ATLException
	{
		int retour = 0;
		if (unit==null || finalUnit==null) {
			String message = "No unit defined for conversion for "+enumeration.getName();
			throw new ATLException(message);
		}
		if (unit.equals(finalUnit)) {
			return (value);
		}
		if (marteIsTypeOf(unit,marte.getConstants().Unit)) {
			Object baseUnitObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_baseUnit);
			Object convFactorObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_convFactor);
			Object offsetFactorObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_offsetFactor);
			if (baseUnitObject instanceof EnumerationLiteral) {
				EnumerationLiteral baseUnit = (EnumerationLiteral) baseUnitObject;
				int tmpValue = value;
				if (offsetFactorObject!=null) {
					if (offsetFactorObject instanceof String) {
						String offsetFactor = (String) offsetFactorObject;
						tmpValue = tmpValue + new Integer(offsetFactor).intValue();
					}
				}
				if (convFactorObject!=null) {
					if (convFactorObject instanceof String) {
						String convFactor = (String) convFactorObject;
						tmpValue = tmpValue * new Integer(convFactor).intValue();
					}
				}
				retour = marteConvertIntegerValue(enumeration, tmpValue,baseUnit, finalUnit);
			}
		}
		
		return retour;
	}
	
	/**
	 * Converts an integer value in unit to a value in finalUnit
	 * @throws ATLException
	 */
	public static int marteConvertIntegerValue(Enumeration enumeration, int value, String unitString, String finalUnitString)
		throws ATLException
	{
		int retour = 0;
		if (unitString==LITERAL_DEFAULT || finalUnitString==LITERAL_DEFAULT) {
			retour = value;
		}
		else {
			EnumerationLiteral unit = marteGetUnit(enumeration, unitString);
			EnumerationLiteral finalUnit = marteGetUnit(enumeration, finalUnitString);
			retour = marteConvertIntegerValue(enumeration, value, unit, finalUnit);
		}
		return retour;
	}

	public static boolean marteCanConvertVSLIntegerValue (ValueSpecification valueSpecification, Enumeration unitKind, String finalUnit) {
		boolean retour = true;
		try {
			marteConvertVSLIntegerValue(valueSpecification, unitKind, finalUnit);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Converts the value (int) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @throws ATLException
	 * @deprecated
	 */	
	public static int marteConvertVSLIntegerValue(ValueSpecification valueSpecification, Enumeration unitKind, String finalUnit)
		throws ATLException
	{
		int retour = 0;
		Log.verboseMessage(ATLTransform.PluginID, "marteConvertVSLIntValue");

		ArrayList<String> valueAttribList = new ArrayList<String>();
		//TODO get "value" from valueAttrib
		valueAttribList.add("value");
		int value = marteGetIntegerValue(valueSpecification, valueAttribList);
		
		ArrayList<String> unitAttribList = new ArrayList<String>();
		//TODO get "unit" from unitAttrib
		unitAttribList.add("unit");
		String unit = marteGetStringValue(valueSpecification, unitAttribList);
		
		Log.debugMessage(ATLTransform.PluginID, "Value : "+value+ " "+unit);
		
		retour = marteConvertIntegerValue(unitKind, value, unit, finalUnit);
		return(retour);
	}
	
	/**
	 * Converts the value (int) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @param valueSpecification ValueSpecification that contains the value to get and convert
	 * @param nfpType NfpType of the value to get and convert
	 * @param finalUnit Name of the unit to convert the value in
	 * @return Value (int) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @throws ATLException
	 */
	public static int marteConvertVSLIntegerValue(ValueSpecification valueSpecification, DataType nfpType, String finalUnit)
		throws ATLException
	{
		int retour = 0;
		Log.verboseMessage(ATLTransform.PluginID, "marteConvertVSLIntValue");

		ArrayList<String> valueAttribList = new ArrayList<String>();
		valueAttribList.add(marteGetValueAttrib(nfpType));
		int value = marteGetIntegerValue(valueSpecification, valueAttribList);
		
		ArrayList<String> unitAttribList = new ArrayList<String>();
		unitAttribList.add(marteGetUnitAttrib(nfpType));
		String unit = marteGetStringValue(valueSpecification, unitAttribList);
		
		Log.debugMessage(ATLTransform.PluginID, "Value : "+value+ " "+unit);
		
		Enumeration unitKind = marteGetUnitKind(nfpType);
		retour = marteConvertIntegerValue(unitKind, value, unit, finalUnit);
		return(retour);
	}

	/*
	 * Double
	 */

	/**
	 * Returns true if a value (double) is corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib Name of the attribute to get if any
	 * @return If a value (double) is corresponding to the valueSpecification
	 */
	public static boolean marteHasDoubleValue (ValueSpecification valueSpecification, ArrayList<String> attribList) {
		boolean retour = true;
		try {
			marteGetDoubleValue(valueSpecification, attribList);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Returns a double corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib NAme of the attribute to get if any
	 * @returna Double value of valueSpecification
	 * @throws ATLException
	 */
	//TODO E.M. Treat Expressions
	public static double marteGetDoubleValue(ValueSpecification valueSpecification, ArrayList<String> attribList) throws ATLException {
		boolean treated = false;
		double retour=0;
		if (valueSpecification instanceof LiteralSpecification) {
			LiteralSpecification literalSpecification = (LiteralSpecification) valueSpecification;
			if (literalSpecification instanceof LiteralString) {
				LiteralString literalString = (LiteralString) literalSpecification;
				treated = true;
				retour = new Double(literalString.getValue()).doubleValue();
			}
			if (literalSpecification instanceof LiteralBoolean) {
				LiteralBoolean literalBoolean = (LiteralBoolean) literalSpecification;
				treated=true;
				retour = literalBoolean.integerValue();
			}
			if (literalSpecification instanceof LiteralDateTime) {
				LiteralDateTime literalDateTime = (LiteralDateTime) literalSpecification;
				treated = true;
				VSLDate v= literalDateTime.getValue();
				retour = (((((v.year*12+v.month)*30+v.dayOfMonth)*24+v.hour)*60+v.min)*60+v.sec)*100+v.centisec;
			}
			if (literalSpecification instanceof LiteralNull) {
				treated=true;
				retour = 0;
			}
			if (literalSpecification instanceof LiteralDefault) {
				treated=true;
				retour = 0;
			}
			if (literalSpecification instanceof LiteralInteger) {
				LiteralInteger literalInteger = (LiteralInteger) literalSpecification;
				treated=true;
				return literalInteger.getValue();
			}
			if (literalSpecification instanceof LiteralUnlimitedNatural) {
				LiteralUnlimitedNatural literalUnlimitedNatural = (LiteralUnlimitedNatural) literalSpecification;
				treated=true;
				return literalUnlimitedNatural.getValue();
			}
			if (literalSpecification instanceof LiteralReal) {
				LiteralReal literalReal = (LiteralReal) literalSpecification;
				treated=true;
				return literalReal.getValue();
			}
		}
		if (valueSpecification instanceof EnumerationSpecification) {
			EnumerationSpecification enumerationSpecification = (EnumerationSpecification) valueSpecification;
			return marteGetDoubleValue(enumerationSpecification.getNameExpression(), attribList);
		}
		//TODO Treat OpaqueExpression
//		if (valueSpecification instanceof Expression) {
			//TODO Treat VariableCallExpression
			//TODO Treat Variable
			//TODO Treat PropertyCallExpression
			//TODO Treat OperationCallExpression
			if (valueSpecification instanceof ConditionalExpression) {
				ConditionalExpression conditionalExpression = (ConditionalExpression) valueSpecification;
				boolean condition = marteGetBooleanValue(conditionalExpression.getConditionExpr(), attribList);
				if (condition) {
					treated=true;
					retour = marteGetDoubleValue(conditionalExpression.getIfTrueExpr(), attribList);
				} else {
					treated=true;
					retour = marteGetDoubleValue(conditionalExpression.getIfFalseExpr(), attribList);
				}
			}
//		}
		if (valueSpecification instanceof IntervalSpecification) {
			IntervalSpecification intervalSpecification = (IntervalSpecification) valueSpecification;
			double min = marteGetDoubleValue(intervalSpecification.getMin(), attribList);
			double max = marteGetDoubleValue(intervalSpecification.getMax(), attribList);
			treated = true;
			retour = (max + min)/2;
		}
		if (valueSpecification instanceof TupleSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetDoubleValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof CollectionSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetDoubleValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof ChoiceSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				treated=true;
				retour = marteGetDoubleValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof TimeExpression) {
			TimeExpression timeExpression = (TimeExpression) valueSpecification;
			if (timeExpression.getExpr()!=null) {
				treated = true;
				retour = marteGetDoubleValue(timeExpression.getExpr(), attribList);
			}
			//TODO treat else case
		}
		if (!treated) {
			String message = "Error in interpreting a value specification as a double expression";
			throw new ATLException(message);
		}
		return retour;
	}
	
	/**
	 * Converts a double value in unit to a value in finalUnit
	 * @param enumeration
	 * @param value
	 * @param unit
	 * @param finalUnit
	 * @return Value converted into finalUnit
	 * @throws ATLException
	 */
	public static double marteConvertDoubleValue(Enumeration enumeration, double value, EnumerationLiteral unit, EnumerationLiteral finalUnit)
		throws ATLException
	{
		double retour = 0;
		if (unit==null || finalUnit==null) {
			String message = "No unit defined for conversion for "+enumeration.getName();
			throw new ATLException(message);
		}
		if (unit.equals(finalUnit)) {
			return (value);
		}
		if (marteIsTypeOf(unit,marte.getConstants().Unit)) {
			Object baseUnitObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_baseUnit);
			Object convFactorObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_convFactor);
			Object offsetFactorObject = marteGetPropertyValue(unit, marte.getConstants().Unit, marte.getConstants().Unit_offsetFactor);
			if (baseUnitObject instanceof EnumerationLiteral) {
				EnumerationLiteral baseUnit = (EnumerationLiteral) baseUnitObject;
				double tmpValue = value;
				if (offsetFactorObject!=null) {
					if (offsetFactorObject instanceof String) {
						String offsetFactor = (String) offsetFactorObject;
						tmpValue = tmpValue + new Double(offsetFactor).doubleValue();
					}
				}
				if (convFactorObject!=null) {
					if (convFactorObject instanceof String) {
						String convFactor = (String) convFactorObject;
						tmpValue = tmpValue * new Double(convFactor).doubleValue();
					}
				}
				retour = marteConvertDoubleValue(enumeration, tmpValue,baseUnit, finalUnit);
			}
		}
		
		return retour;
	}
	
	/**
	 * Converts a double value in unit to a value in finalUnit
	 * @param enumeration
	 * @param value
	 * @param unitString
	 * @param finalUnitString
	 * @return Value converted into finalUnit
	 * @throws ATLException
	 */
	public static double marteConvertDoubleValue(Enumeration enumeration, double value, String unitString, String finalUnitString)
		throws ATLException
	{
		double retour = 0;
		if (unitString.equals(LITERAL_DEFAULT) || finalUnitString.equals(LITERAL_DEFAULT)) {
			retour = value;
		}
		else {
			EnumerationLiteral unit = marteGetUnit(enumeration, unitString);
			EnumerationLiteral finalUnit = marteGetUnit(enumeration, finalUnitString);
			retour = marteConvertDoubleValue(enumeration, value, unit, finalUnit);
		}
		return (retour);
	}

	/**
	 * 
	 * @param valueSpecification
	 * @param unitKind
	 * @param finalUnit
	 * @return
	 * @deprecated
	 */
	public static boolean marteCanConvertVSLDoubleValue (ValueSpecification valueSpecification, Enumeration unitKind, String finalUnit) {
		boolean retour = true;
		try {
			marteConvertVSLDoubleValue(valueSpecification, unitKind, finalUnit);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Converts the value (double) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @param element
	 * @param stereotype
	 * @param property
	 * @param finalUnit
	 * @return Value converted into finalUnit
	 * @throws ATLException
	 * @deprecated
	 */
	public static double marteConvertVSLDoubleValue(ValueSpecification valueSpecification, Enumeration unitKind, String finalUnit)
		throws ATLException
	{
		double retour = 0;
		Log.verboseMessage(ATLTransform.PluginID, "marteConvertVSLDoubleValue");

		ArrayList<String> valueAttribList = new ArrayList<String>();
		//TODO get "value" from valueAttrib
		valueAttribList.add("value");
		double value = marteGetDoubleValue(valueSpecification, valueAttribList);
			
		ArrayList<String> unitAttribList = new ArrayList<String>();
		//TODO get "unit" from unitAttrib
		unitAttribList.add("unit");
		String unit = marteGetStringValue(valueSpecification, unitAttribList);
			
		Log.debugMessage(ATLTransform.PluginID, "Value : "+value+ " "+unit);
			
		retour = marteConvertDoubleValue(unitKind, value, unit, finalUnit);
		return(retour);
	}
	
	/**
	 * Return true if the value of valueSpecification can be converted in finalUnit
	 * @param valueSpecification ValueSpecification that contains the value to get and convert
	 * @param nfpType Type of the value to get and convert
	 * @param finalUnit Unit to convert the value in
	 * @return If the value of valueSpecification can be converted in finalUnit
	 */
	public static boolean marteCanConvertVSLDoubleValue (ValueSpecification valueSpecification, DataType nfpType, String finalUnit) {
		boolean retour = true;
		try {
			marteConvertVSLDoubleValue(valueSpecification, nfpType, finalUnit);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Converts the value (double) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @param valueSpecification ValueSpecification that contains the value to get and convert
	 * @param nfpType NfpType of the value to get and convert
	 * @param finalUnit Name of the unit to convert the value in
	 * @return Value (double) of the stereotype attribute (property) of element in unit to a value in finalUnit
	 * @throws ATLException
	 */
	public static double marteConvertVSLDoubleValue(ValueSpecification valueSpecification, DataType nfpType, String finalUnit)
		throws ATLException
	{
		double retour = 0;
		Log.verboseMessage(ATLTransform.PluginID, "marteConvertVSLDoubleValue");

		ArrayList<String> valueAttribList = new ArrayList<String>();
		valueAttribList.add(marteGetValueAttrib(nfpType));
		double value = marteGetDoubleValue(valueSpecification, valueAttribList);
		
		ArrayList<String> unitAttribList = new ArrayList<String>();
		unitAttribList.add(marteGetUnitAttrib(nfpType));
		String unit = marteGetStringValue(valueSpecification, unitAttribList);
		
		Log.debugMessage(ATLTransform.PluginID, "Value : "+value+ " "+unit);
		
		Enumeration unitKind = marteGetUnitKind(nfpType);
		retour = marteConvertDoubleValue(unitKind, value, unit, finalUnit);
		return(retour);
	}

	/*
	 * String
	 */

	/**
	 * Returns true if a value (String) is corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib Name of the attribute to get if any
	 * @return If a value (String) is corresponding to the valueSpecification
	 */
	public static boolean marteHasStringValue (ValueSpecification valueSpecification, ArrayList<String> attribList) {
		boolean retour = true;
		try {
			marteGetStringValue(valueSpecification, attribList);
			retour = true;
		}
		catch (Exception exception) {
			retour = false;
		}
		return retour;
	}
	
	/**
	 * Returns a String corresponding to the valueSpecification
	 * @param valueSpecification
	 * @param attrib NAme of the attribute to get if any
	 * @returna String value of valueSpecification
	 * @throws ATLException
	 */
	//TODO E.M. Treat Expressions
	public static String marteGetStringValue(ValueSpecification valueSpecification, ArrayList<String> attribList)
			throws ATLException {
		String retour=null;
		if (valueSpecification instanceof LiteralSpecification) {
			LiteralSpecification literalSpecification = (LiteralSpecification) valueSpecification;
			if (literalSpecification instanceof LiteralString) {
				LiteralString literalString = (LiteralString) literalSpecification;
				
				retour = literalString.getValue();
			}
			if (literalSpecification instanceof LiteralBoolean) {
				LiteralBoolean literalBoolean = (LiteralBoolean) literalSpecification;
				retour = new Boolean(literalBoolean.booleanValue()).toString();
			}
			if (literalSpecification instanceof LiteralDateTime) {
				LiteralDateTime literalDateTime = (LiteralDateTime) literalSpecification;
				retour = literalDateTime.getValue().toString();
			}
			if (literalSpecification instanceof LiteralNull) {
				retour = LITERAL_NULL;
			}
			if (literalSpecification instanceof LiteralDefault) {
				retour = LITERAL_DEFAULT;
			}
			if (literalSpecification instanceof LiteralInteger) {
				LiteralInteger literalInteger = (LiteralInteger) literalSpecification;
				retour = new Integer(literalInteger.integerValue()).toString();
			}
			if (literalSpecification instanceof LiteralUnlimitedNatural) {
				LiteralUnlimitedNatural literalUnlimitedNatural = (LiteralUnlimitedNatural) literalSpecification;
				retour = new Integer(literalUnlimitedNatural.getValue()).toString();
			}
			if (literalSpecification instanceof LiteralReal) {
				LiteralReal literalReal = (LiteralReal) literalSpecification;
				retour = new Double(literalReal.getValue()).toString();
			}
		}
		if (valueSpecification instanceof EnumerationSpecification) {
			EnumerationSpecification enumerationSpecification = (EnumerationSpecification) valueSpecification;
			retour = enumerationSpecification.getName();
		}
		//TODO Treat OpaqueExpression
//		if (valueSpecification instanceof Expression) {
			//TODO Treat VariableCallExpression
			//TODO Treat Variable
			//TODO Treat PropertyCallExpression
			//TODO Treat OperationCallExpression
			if (valueSpecification instanceof ConditionalExpression) {
				ConditionalExpression conditionalExpression = (ConditionalExpression) valueSpecification;
				boolean condition = marteGetBooleanValue(conditionalExpression.getConditionExpr(), attribList);
				if (condition) {
					retour = marteGetStringValue(conditionalExpression.getIfTrueExpr(), attribList);
				} else {
					retour = marteGetStringValue(conditionalExpression.getIfFalseExpr(), attribList);
				}
			}
//		}
		if (valueSpecification instanceof IntervalSpecification) {
			IntervalSpecification intervalSpecification = (IntervalSpecification) valueSpecification;
			retour = "";
			if (intervalSpecification.isIsLowerOpen()) {
				retour = retour + "]";
			} else {
				retour = retour + "[";
			}
			retour = retour + marteGetStringValue(intervalSpecification.getMin(), attribList);
			retour = retour + ",";
			retour = retour + marteGetStringValue(intervalSpecification.getMax(), attribList);
			if (intervalSpecification.isIsUpperOpen()) {
				retour = retour + "[";
			} else {
				retour = retour + "]";
			}
		}
		if (valueSpecification instanceof TupleSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				retour = marteGetStringValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof CollectionSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				retour = marteGetStringValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof ChoiceSpecification) {
			ValueSpecification newValueSpecification = marteGetValueSpecification(valueSpecification,attribList);
			if (newValueSpecification!=null) {
				retour = marteGetStringValue(newValueSpecification,attribList);
			}
		}
		if (valueSpecification instanceof TimeExpression) {
			TimeExpression timeExpression = (TimeExpression) valueSpecification;
			if (timeExpression.getExpr()!=null) {
				retour = marteGetStringValue(timeExpression.getExpr(), attribList);
			}
			//TODO treat else case
		}
		if (retour==null) {
			String message = "Error in interpreting a value specification as a String expression";
			throw new ATLException(message);
		}
		return retour;
	}

}
