/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>
 */
package com.thalesgroup.vslview.rsa;

import java.util.ArrayList;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.StructuralFeature;
import org.eclipse.uml2.uml.ValueSpecification;


import com.cea.nfp.parsers.modelgenerator.VSLFullParser;
import com.thalesgroup.nfp.rsa.facade.RSAModelFacade;
import com.thalesgroup.vslviewer.extension.VSLSelectionConverter;

/**
 * The VSLSelectionConverter for RSA.
 * @author T0081227 Francois NIZOU - 18 oct. 07
 * 
 */
public class RSAConverter implements VSLSelectionConverter {

	/**
	 * Compute a VSL object from the selection.
	 */
	public ValueSpecification convert(ISelection selection) {
		ValueSpecification vsl = null;

		if (selection instanceof TreeSelection) {
			TreeSelection treeSel = (TreeSelection) selection;
			Object firstElement = treeSel.getFirstElement();

			if (firstElement instanceof Element) {
				vsl = getVSLFromElement((Element) firstElement);
			}

			else if (firstElement instanceof IAdaptable) {
				IAdaptable adaptable = (IAdaptable) firstElement;
				Object adapted = adaptable.getAdapter(Element.class);
				if (adapted != null && adapted instanceof Element) {
					vsl = getVSLFromElement((Element) adapted);
				}
			}

		} else if (selection instanceof IStructuredSelection) {
			IStructuredSelection structSel = (IStructuredSelection) selection;
			Object firstElement = structSel.getFirstElement();

			if (firstElement instanceof IAdaptable) {
				IAdaptable adaptable = (IAdaptable) firstElement;
				Object adapted = adaptable.getAdapter(Element.class);
				if (adapted != null && adapted instanceof Element) {
					vsl = getVSLFromElement((Element) adapted);
				}
			}
		}
		return vsl;
	}

	/**
	 * Compute a VSL object from an UML Element
	 * @param e an UML element. Only constraint, slot and property are handled.
	 * @return
	 */
	public ValueSpecification getVSLFromElement(Element e) {
		RSAModelFacade facade = new RSAModelFacade(e.getModel(), e);

		String vslStr = null;
		DataType datatype = null;

		if (e instanceof Constraint) {
			Constraint constraint = (Constraint) e;
			ValueSpecification specification = constraint.getSpecification();
			if (specification instanceof OpaqueExpression) {
				vslStr = getVSLStr((OpaqueExpression) specification);
				ArrayList<DataType> dataTypesByName = facade
						.getDataTypesByName("Boolean");
				if (dataTypesByName != null && dataTypesByName.size() > 0)
					datatype = dataTypesByName.get(0);
				else
					return null;
			}
		} else if (e instanceof Slot) {
			Slot slot = (Slot) e;
			EList values = slot.getValues();
			if (values != null && values.size() > 0) {
				Object specification = values.get(0);
				if (specification instanceof OpaqueExpression) {
					vslStr = getVSLStr((OpaqueExpression) specification);

				}
				StructuralFeature definingFeature = slot.getDefiningFeature();
				if (definingFeature instanceof Property) {
					Property p = (Property) definingFeature;
					DataType typeof = facade.typeof(p);
					if (typeof != null)
						datatype = typeof;
					else
						return null;
				}
			}
		} else if (e instanceof Property) {
			Property property = (Property) e;
			vslStr = property.getDefault();
			datatype = facade.typeof(property);

		}

		if (vslStr != null) {

			try {
				ValueSpecification vsl = VSLFullParser.parse(vslStr, datatype,
						facade);
				return vsl;
			} catch (Exception e1) {
				return null;
			}
		}
		return null;
	}

	/**
	 * Get the VSL String from an opaque expression.
	 * @param oe
	 * @return
	 */
	public String getVSLStr(OpaqueExpression oe) {
		Object[] toArray = oe.getLanguages().toArray();
		int index = -1;
		for (int i = 0; i < toArray.length; i++) {
			if (toArray[i].equals("VSL")) {
				index = i;
				break;
			}
		}
		if (index >= 0) {
			if (index < oe.getBodies().size())
				return oe.getBodies().get(index).toString();
		}

		return null;
	}
}
