/**
 * <copyright>
 * THALES MARTE TO RAPIDRMA (Copyright (c) THALES 2008 All rights reserved) is free software; you can redistribute it and/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html 
 *  
 * THALES MARTE TO RAPIDRMA is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details.
 *
 * </copyright>
 * 
 */

package com.thalesgroup.java.log;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;

/**
 * Class used to manage debugging, verbose, warning and error messages
 * The ability to display messages is parametrized using debug, verbose, warning and error variable
 * and the pluginIDs that characterized the context.
 * The "ALL" pluginID displays all messages.
 * @author maes
 *
 */
public class Log{

	public static final Integer PROFILE_USER = 0;
	public static final Integer PROFILE_DEMONSTRATOR = 1;
	public static final Integer PROFILE_DEVELOPPER = 2;
	public static final Integer PROFILE_DEBUGGER = 3;
	
	protected static boolean DEBUG = false;
	protected static boolean VERBOSE = false;
	protected static boolean DEMO = true;
	protected static boolean WARNING = true;
	protected static boolean ERROR = true;
	
	public static Integer PROFILE = PROFILE_USER;
	
	protected static boolean DISPLAY_CONSOLE = true;
	protected static boolean DISPLAY_ERRORLOG = true;
	protected static boolean DISPLAY_FILE = false;
	
	protected static boolean INITIALIZED_FILE = false;
	protected static PrintWriter LOG_FILE;
	protected static String LOG_FILE_PATH = "com.thalesgroup.java.log";
	
	public static LogConsole  LOG_CONSOLE = new LogConsole();
	
	/**
	 * File into which display messages regarding Log options
	 */
	public static IFile file;
	
	/**
	 * plugin ID to take into account all plugin messages
	 */
	public static String ALL = "ALL"; 
	
	/**
	 * List of pluginIDs for which displaying messages
	 */
	private static List<String> pluginIDs = new ArrayList<String>();
	
	/**
	 * Initialization
	 */
	private static boolean INITIALIZED=false;
	
	/**
	 * Initialization
	 */
	private static void init() {
		if (!INITIALIZED) {
			INITIALIZED = true;
			int profile = PROFILE_USER;
			boolean console = false;
			boolean errorLog = false;
			boolean file = false;
			String filePath="";
			if (Activator.getDefault().getPreferenceStore() != null) {
				if (Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_PROFILE) != null 
						&&
						Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_PROFILE).length() != 0) {
					Log.debugMessage(Activator.PLUGIN_ID,"Log Profile Preference : "+Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_PROFILE));
					profile = new Integer(Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_PROFILE)).intValue();
				}
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"The Log Preference Store Profile information has not been loaded correctly");
				}
				console = Activator.getDefault().getPreferenceStore().getBoolean(LogPreferencePage.LOG_DISPLAY_CONSOLE);
				errorLog = Activator.getDefault().getPreferenceStore().getBoolean(LogPreferencePage.LOG_DISPLAY_ERRORLOG);
				file = Activator.getDefault().getPreferenceStore().getBoolean(LogPreferencePage.LOG_DISPLAY_FILE);
				if (file) {
					if (Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_FILE_DIRECTORY) != null
							&&
							Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_FILE_NAME) != null) {
						filePath = Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_FILE_DIRECTORY) 
									+ "\\"
									+ Activator.getDefault().getPreferenceStore().getString(LogPreferencePage.LOG_FILE_NAME);
					}
					else {
						Log.warningMessage(Activator.PLUGIN_ID,"The Log Preference Store File information has not been loaded correctly");
					} 
				}
				else {
					Log.warningMessage(Activator.PLUGIN_ID,"The Log Preference Store has not been loaded correctly");
				}
			}
			config(profile, console, errorLog, file, filePath);
			if (file) {
				initFile(filePath);
			}
		}
	}
	
	/**
	 * End of log
	 */
	public static void close() {
		closeFile();
	}
	
	protected static void closeFile() {
		if (INITIALIZED_FILE) {
			LOG_FILE.close();
			INITIALIZED_FILE = false;
		}
	}
	
	/**
	 * File initialization
	 */
	protected static boolean initFile(String logFilePath) {
		if (! INITIALIZED_FILE && ! logFilePath.equals("")) {
			try {
				LOG_FILE_PATH = logFilePath;
				LOG_FILE = new PrintWriter(logFilePath);
				INITIALIZED_FILE = true;
			}
			catch (Exception exception) {
				DISPLAY_FILE = false;
				errorMessage(Activator.PLUGIN_ID, "Impossible to create the file "+LOG_FILE_PATH, exception);
			}
			return true;
		}
		return false;
	}
	protected static boolean initFile() {
		return initFile(LOG_FILE_PATH);
	}
	
	
	/**
	 * Configuration
	 */
	public static void config(int mode) {
		if (mode == PROFILE_USER) { Log.user(); }
		if (mode == PROFILE_DEMONSTRATOR) { Log.demonstrator(); }
		if (mode == PROFILE_DEVELOPPER) { Log.developper(); }
		if (mode == PROFILE_DEBUGGER) { Log.debugger(); }
	}
	
	/**
	 * Configuration
	 */
	public static void config(int mode, String logFilePath) {
		config(mode);
		LOG_FILE_PATH = logFilePath;
		DISPLAY_FILE = true;
		if (DISPLAY_CONSOLE) {
			ConsolePlugin.getDefault().getConsoleManager().addConsoles( new IConsole[] { LOG_CONSOLE } );
		}
	}
	
	/**
	 * Configuration
	 */
	public static void config(int mode, boolean console, boolean errorLog, boolean file, String logFilePath) {
		config(mode, logFilePath);
		DISPLAY_CONSOLE = console;
		DISPLAY_ERRORLOG = errorLog;
		DISPLAY_FILE = file;
	}
	
	public static void setDisplayConsole(boolean set){
		DISPLAY_CONSOLE = set;
		if (set) {
			ConsolePlugin.getDefault().getConsoleManager().addConsoles( new IConsole[] { LOG_CONSOLE } );
		}
		else {
			LOG_CONSOLE.destroy();
		}
	}
	public static void setDisplayErrorLog(boolean set){
		DISPLAY_ERRORLOG = set;
	}
	public static void setDisplayFile(boolean set){
		DISPLAY_FILE = set;
		if (set=true) {
			if (INITIALIZED_FILE) {
				closeFile();
			}
			initFile();
		}
		else {
			closeFile();
		}
	}
	public static void setFilePath(String directoryPath, String fileName){
		LOG_FILE_PATH = directoryPath+'\\'+fileName;
		if (INITIALIZED_FILE) {
			closeFile();
		}
		initFile(LOG_FILE_PATH);
	}
	public static void setFilePath(String filePath){
		LOG_FILE_PATH = filePath;
		if (INITIALIZED_FILE) {
			closeFile();
		}
		initFile(LOG_FILE_PATH);
	}
	
	/**
	 * Configuration for debugging
	 */
	public static void debugger() {
		addModule("ALL");
		PROFILE = PROFILE_DEBUGGER;
		DEBUG = true;
		VERBOSE = true;
		DEMO = false;
		WARNING = true;
		ERROR = true;
	}
	
	/**
	 * Configuration for developping
	 */
	public static void developper() {
		addModule("ALL");
		PROFILE = PROFILE_DEVELOPPER;
		DEBUG = false;
		VERBOSE = true;
		DEMO=true;
		WARNING = true;
		ERROR = true;
	}
	
	/**
	 * Configuration for executing
	 */
	public static void user() {
		addModule("ALL");
		PROFILE = PROFILE_USER;
		DEBUG = false;
		VERBOSE = false;
		DEMO=false;
		WARNING = true;
		ERROR = true;
	}
	
	/**
	 * Configuration for demonstrating
	 */
	public static void demonstrator() {
		addModule("ALL");
		PROFILE = PROFILE_DEMONSTRATOR;
		DEBUG = false;
		VERBOSE = false;
		DEMO=true;
		WARNING = true;
		ERROR = true;
	}
	
	/**
	 * Getter to know if the DEBUG displaying messages function is enabled
	 */
	static public boolean isDEBUGEnabled() {
		init();
		return DEBUG;
	}
	
	/**
	 * Getter to know if the VERBOSE displaying messages function is enabled
	 */
	static public boolean isVERBOSEEnabled() {
		init();
		return VERBOSE;
	}
	
	/**
	 * Getter to know if the DEMO displaying messages function is enabled
	 */
	static public boolean isDEMOEnabled() {
		init();
		return DEMO;
	}
	
	/**
	 * Getter to know if the WARNING displaying messages function is enabled
	 */
	static public boolean isWARNINGEnabled() {
		init();
		return WARNING;
	}
	
	/**
	 * Getter to know if the DEBUG displaying messages function is enabled
	 */
	static public boolean isERROREnabled() {
		init();
		return ERROR;
	}
	
	/**
	 * Add a pluginID for which displaying messages
	 * @param pluginID String corresponding to the context
	 */
	public static void addModule(String pluginID) {
		verboseMessage(Activator.PLUGIN_ID,"Add a module for which displaying messages : "+pluginID);
		if (pluginIDs.contains(ALL)) {
			pluginIDs.remove(pluginID);
			debugMessage(Activator.PLUGIN_ID,"Remove module "+ALL);
		}
		if (! pluginIDs.contains(pluginID)) {
			pluginIDs.add(pluginID);
		}
	}

	/**
	 * Remove a pluginID for which no more displaying messages
	 * @param pluginID String corresponding to the context
	 */
	public static void removeModule(String pluginID) {
		verboseMessage(Activator.PLUGIN_ID,"Remove a module for which no more displaying messages : "+pluginID);
		if (pluginIDs.contains(ALL)) {
			pluginIDs.remove(pluginID);
			debugMessage(Activator.PLUGIN_ID,"Remove module "+ALL);
		}
		if (pluginIDs.contains(pluginID)) {
			pluginIDs.remove(pluginID);
		}
		else {
			warningMessage(ResourcesPlugin.getPlugin().toString(),"Trying to remove a non-existant pluginID");
		}
	}

	/**
	 * Remove pluginIDs for which no more displaying messages
	 */
	public static void removeModules() {
		verboseMessage(Activator.PLUGIN_ID,"Remove pluginIDs for which no more displaying messages");
		pluginIDs.clear();
	}
	
	/**
	 * Displays message on an Eclipse console
	 * @param message
	 */
	protected static void displayOnConsole(String message, int severity) {
		if (DISPLAY_CONSOLE) {
			try {
				init();
				LOG_CONSOLE.display(message, severity);
//			System.out.flush();
//			System.out.println(message);
			}
			catch (Exception exception) {
				setDisplayConsole(false);
				String errorMessage = "Error writing in the Log Console : "+message.toString();
				Log.errorMessage(Activator.PLUGIN_ID, errorMessage, exception);
			}
		}
	}
	
	/**
	 * Displays message in a file
	 * @param message
	 */
	protected static void displayInFile(String message, int severity) {
		if (DISPLAY_FILE) {
			init();
			LOG_FILE.println(message);
			LOG_FILE.flush();
		}
	}
	
	/**
	 * Displays message on an Eclipse ErrorLog
	 * @param severity
	 * @param pluginID
	 * @param code
	 * @param message
	 * @param exception
	 */
	protected static void displayOnErrorLog(int severity, String pluginID, int code, String message, Throwable exception) {
		if (DISPLAY_ERRORLOG) {
			init();
			ResourcesPlugin.getPlugin().getLog().log(new Status(severity,pluginID,code,message,exception));
		}
	}
	protected static void displayOnErrorLog(int severity, String pluginID, int code, String message) {
		displayOnErrorLog(severity, pluginID, code, message, null);
	}
	
	/**
	 * Display a message for debugging
	 * The message is displayed if DEBUG is true and pluginIDs contains pluginID
	 * @param pluginID String corresponding to the context
	 * @param message Message to display
	 */
	public static void debugMessage(String pluginID, String message) {
		init();
		String debugMessage = "DEBUG / "+pluginID+" / "+message;
		if (pluginIDs.contains(ALL) || pluginIDs.contains(pluginID)) { 
			if (DEBUG) {
				displayOnConsole(debugMessage, Status.OK);
				displayInFile(debugMessage, Status.OK);
			}
		}
	}

	/**
	 * Display a message for developping
	 * The message is displayed if VERBOSE is true and pluginIDs contains pluginID
	 * @param pluginID String corresponding to the context
	 * @param message Message to display
	 */	public static void verboseMessage(String pluginID, String message) {
			init();
			String verboseMessage = "VERBOSE / "+pluginID+" / "+message;
			if (pluginIDs.contains(ALL) || pluginIDs.contains(pluginID)) {
				if (VERBOSE) {
					displayOnConsole(verboseMessage, Status.OK);
					displayInFile(verboseMessage,Status.OK);
				}
			}
	 }

	/**
	 * Display a message for demonstration
	 * The message is displayed if DEMO is true and pluginIDs contains pluginID
	 * @param pluginID String corresponding to the context
	 * @param message Message to display
	 */	public static void demoMessage(String pluginID, String message) {
			init();
			String demoMessage = "DEMO / "+message;
			if (pluginIDs.contains(ALL) || pluginIDs.contains(pluginID)) {
				if (DEMO) {
					displayOnConsole(demoMessage, Status.INFO);
					displayInFile(demoMessage, Status.INFO);
					displayOnErrorLog(Status.INFO, Activator.PLUGIN_ID, Status.OK, message);
				}
			}
	 }

	/**
	 * Display a message for to warn the user for warnings
	 * The message is displayed if WARNING is true and pluginIDs contains pluginID
	 * @param pluginID String corresponding to the context
	 * @param message Message to display
	 */
	public static void warningMessage(String pluginID, String message) {
		init();
		warningMessage(pluginID,message,null);
	}
	public static void warningMessage(String pluginID, String message, Exception exception) {
		init();
		String warningMessage = "WARNING / "+pluginID+" / "+message;
		if (WARNING) { 
			if (VERBOSE) {
				displayOnConsole(warningMessage, Status.WARNING);
			}
			displayInFile(warningMessage, Status.WARNING);
			displayOnErrorLog(Status.WARNING,pluginID,Status.OK,message,exception);
		}
	}

	/**
	 * Display a message for to warn the user for errors
	 * The message is displayed if ERROR is true and pluginIDs contains pluginID
	 * @param pluginID String corresponding to the context
	 * @param message Message to display
	 */
	public static void errorMessage(String pluginID, String message) {
		init();
		errorMessage(pluginID,message,null);
	}
	public static void errorMessage(String pluginID, String message, Exception exception) {
		init();
		String errorMessage = "ERROR / "+pluginID+" / "+message;
		if (ERROR) {
			if (VERBOSE) { 
				displayOnConsole(errorMessage,Status.ERROR);
			}
			displayInFile(errorMessage,Status.ERROR);
			displayOnErrorLog(Status.ERROR,pluginID,Status.OK,message,exception);
		}
	}
}
