/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>
 */
package com.thalesgroup.vslviewer.treeObject;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.DurationObservation;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralNull;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Observation;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.TimeObservation;
import org.eclipse.uml2.uml.ValueSpecification;

import com.thalesgroup.marte.vsl.VSLDate;

import VSL.ChoiceSpecification;
import VSL.CollectionSpecification;
import VSL.ConditionalExpression;
import VSL.DurationExpression;
import VSL.EnumerationSpecification;
import VSL.InstantExpression;
import VSL.IntervalSpecification;
import VSL.JitterExpression;
import VSL.LiteralDateTime;
import VSL.LiteralDefault;
import VSL.LiteralReal;
import VSL.ObsCallExpression;
import VSL.OperationCallExpression;
import VSL.PropertyCallExpression;
import VSL.TupleItemValue;
import VSL.TupleSpecification;
import VSL.Variable;
import VSL.VariableCallExpression;

/**
 * This class is the data model of the treeview. For display reason, this of the VSL model tree.
 * A VSL treenode as a label to be displayed and a list of childs. It does also contain the UML Element it represent.
 * @author T0081227 Francois NIZOU - 18 oct. 07 
 *
 */
public class VSLTreeNode {

	protected Element element;

	public Element getElement() {
		return element;
	}

	public boolean isLeaf() {
		return (childs == null || childs.size() == 0);
	}

	protected ArrayList<VSLTreeNode> childs = null;

	public ArrayList<VSLTreeNode> getChilds() {
		return childs;
	}

	protected String label;

	public String getLabel() {
		return label;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	// Literals

	public VSLTreeNode(LiteralDefault vsl) {
		label = "-";
		element = vsl;
	}

	public VSLTreeNode(LiteralNull vsl) {
		label = "null";
		element = vsl;
	}

	public VSLTreeNode(LiteralInteger vsl) {
		label = "Integer: " + vsl.integerValue();
		element = vsl;
	}

	public VSLTreeNode(LiteralString vsl) {
		label = "String: " + vsl.stringValue();
		element = vsl;
	}

	public VSLTreeNode(LiteralReal vsl) {
		label = "Real: " + vsl.getValue();
		element = vsl;
	}

	public VSLTreeNode(LiteralBoolean vsl) {
		label = "Boolean: " + vsl.booleanValue();
		element = vsl;
	}

	public VSLTreeNode(LiteralDateTime vsl) {
		VSLDate date = vsl.getValue();
		label = "DateTime: " + date;
		element = vsl;
	}

	public VSLTreeNode(LiteralUnlimitedNatural vsl) {
		String val = (vsl.unlimitedValue() == -1) ? "*" : vsl.unlimitedValue()
				+ "";
		label = "UnlimitedNatural: " + val;
		element = vsl;
	}

	public VSLTreeNode(EnumerationSpecification vsl) {
		EnumerationLiteral enumLiteral = vsl.getEnumLiteral();
		Enumeration enumeration = enumLiteral.getEnumeration();
		label = "Enumeration " + enumeration.getName() + ": "
				+ enumLiteral.getName();
		element = vsl;
	}

	// composite value

	public VSLTreeNode(IntervalSpecification vsl) {
		String lower = (vsl.isIsLowerOpen()) ? "]" : "[";
		String upper = (vsl.isIsUpperOpen()) ? "[" : "]";
		label = "Interval Specification: " + lower + ".." + upper;

		childs = new ArrayList<VSLTreeNode>();
		VSLTreeNode node = VSLTreeNode.create(vsl.getMin());
		node.setLabel("min=" + node.getLabel());
		childs.add(node);

		node = VSLTreeNode.create(vsl.getMax());
		node.setLabel("max=" + node.getLabel());
		childs.add(node);
		element = vsl;
	}

	public VSLTreeNode(CollectionSpecification vsl) {
		label = "Collection Specification";
		childs = new ArrayList<VSLTreeNode>();

		VSLTreeNode node = null;
		ValueSpecification[] vals = (ValueSpecification[]) vsl.getItemValue()
				.toArray(new ValueSpecification[] {});
		for (int i = 0; i < vals.length; i++) {
			node = VSLTreeNode.create(vals[i]);
			node.setLabel("item " + i + "=" + node.getLabel());
			childs.add(node);
		}
		element = vsl;
	}

	public VSLTreeNode(TupleSpecification vsl) {
		label = "Tuple Specification";
		if (vsl.getType() != null)
			label += " " + vsl.getType().getName();

		childs = new ArrayList<VSLTreeNode>();

		VSLTreeNode node = null;
		TupleItemValue[] vals = (TupleItemValue[]) vsl.getTupleItem().toArray(
				new TupleItemValue[] {});

		for (TupleItemValue value : vals) {
			node = VSLTreeNode.create(value.getItemValue());
			node.setLabel(value.getTupleAttribute().getName() + " = "
					+ node.getLabel());
			childs.add(node);
		}
		element = vsl;
	}

	public VSLTreeNode(ChoiceSpecification vsl) {
		label = "Choice Specification:";

		childs = new ArrayList<VSLTreeNode>();
		VSLTreeNode node = VSLTreeNode.create(vsl.getValue());
		node.setLabel("chosen alternative: "
				+ vsl.getChoiceAttribute().getName() + " = " + node.getLabel());

		childs.add(node);
	}

	// Expressions

	public VSLTreeNode(Variable vsl) {
		String name = vsl.getName();

		if (vsl.getContext() != null && vsl.getContext().getName() != null
				&& vsl.getContext().getName().length() != 0)
			name = vsl.getContext().getName() + "." + name;

		label = vsl.getDirection() + " Variable: " + name;

		childs = new ArrayList<VSLTreeNode>();

		VSLTreeNode node = new VSLTreeNode(vsl.getDatatype());
		node.setLabel("Datatype: " + node.getLabel());
		childs.add(node);

		ValueSpecification initExpression = vsl.getInitExpression();

		if (initExpression != null) {
			node = VSLTreeNode.create(initExpression);
			node.setLabel("initial expression: " + node.getLabel());
			childs.add(node);
		}

		element = vsl;

	}

	public VSLTreeNode(VariableCallExpression vsl) {
		label = "Variable Call Expression";
		childs = new ArrayList<VSLTreeNode>();
		childs.add(new VSLTreeNode(vsl.getDefiningVariable()));
		element = vsl;
	}

	public VSLTreeNode(PropertyCallExpression vsl) {
		label = "Property Call Expression";
		childs = new ArrayList<VSLTreeNode>();
		childs.add(new VSLTreeNode(vsl.getDefiningProperty()));
		element = vsl;
	}

	public VSLTreeNode(ConditionalExpression vsl) {
		label = "Conditional Expression";

		childs = new ArrayList<VSLTreeNode>();
		VSLTreeNode node = VSLTreeNode.create(vsl.getConditionExpr());
		node.setLabel("condition" + node.getLabel());
		childs.add(node);

		node = VSLTreeNode.create(vsl.getIfTrueExpr());
		node.setLabel("consequence" + node.getLabel());
		childs.add(node);

		node = VSLTreeNode.create(vsl.getIfTrueExpr());
		node.setLabel("alternant" + node.getLabel());
		childs.add(node);

		element = vsl;
	}

	public VSLTreeNode(OperationCallExpression vsl) {
		element = vsl;
		label = "Operation Call Expression";

		childs = new ArrayList<VSLTreeNode>();
		ValueSpecification[] args = (ValueSpecification[]) vsl.getArgument()
				.toArray(new ValueSpecification[] {});
		childs.add(new VSLTreeNode(vsl.getDefiningOperation(), args));
	}

	// UML

	public VSLTreeNode(DataType datatype) {
		label = datatype.getName();
		element = datatype;
	}

	public VSLTreeNode(Property property) {
		Namespace ptr = property.getNamespace();
		String fqn = property.getName();
		while (!(ptr instanceof Model) && ptr != null) {
			fqn = ptr.getName() + ".fqn";
			ptr = ptr.getNamespace();
		}
		label = "Property: " + fqn + " (" + property.getType().getName() + ")";
	}

	public VSLTreeNode(Operation operation, ValueSpecification[] args) {
		this.element = operation;

		childs = new ArrayList<VSLTreeNode>();

		Parameter returnResult = operation.getReturnResult();

		if (returnResult.getType() instanceof DataType) {
			DataType datatype = (DataType) returnResult.getType();
			VSLTreeNode node = new VSLTreeNode(datatype);
			node.setLabel("return " + node.getLabel());
			childs.add(node);
		}

		VSLTreeNode node = VSLTreeNode.create(args[0]);
		node.setLabel("receiver object = " + node.getLabel());
		childs.add(node);

		ArrayList<Parameter> ownedParameters = new ArrayList<Parameter>(
				operation.getOwnedParameters());

		ownedParameters.remove(returnResult);

		for (int i = 1; i < args.length; i++) {
			node = VSLTreeNode.create(args[i]);
			Parameter param = (Parameter) ownedParameters.get(i - 1);
			String paramStr = (param.getName() == null) ? "" : param.getName()
					+ "=";
			node.setLabel("parameter " + paramStr + node.getLabel());
			childs.add(node);
		}
		label = "operation " + operation.getName();

	}

	protected VSLTreeNode(Observation observation) {
		element = observation;

		if (observation instanceof TimeObservation)
			label = "Time Observation " + observation.getName();

		if (observation instanceof DurationObservation)
			label = "Duration Observation " + observation.getName();
	}

	// TimeExpression

	public VSLTreeNode(InstantExpression vsl) {
		element = vsl;
		label = "Instant Expression";

		childs = new ArrayList<VSLTreeNode>();
		EList obsExpr = vsl.getObsExpr();

		if (obsExpr.size() == 1) {
			ValueSpecification instant = (ValueSpecification) obsExpr.get(0);
			VSLTreeNode node = VSLTreeNode.create(instant);
			childs.add(node);
		} else if (obsExpr.size() == 2) {
			label += label + " " + "(instant + Duration)";
			ValueSpecification instant = (ValueSpecification) obsExpr.get(0);
			ValueSpecification duration = (ValueSpecification) obsExpr.get(0);

			VSLTreeNode node = VSLTreeNode.create(instant);
			childs.add(node);

			node = VSLTreeNode.create(duration);
			childs.add(node);
		}

	}

	public VSLTreeNode(DurationExpression vsl) {
		element = vsl;
		label = "Duration Expression";

		childs = new ArrayList<VSLTreeNode>();
		EList obsExpr = vsl.getObsExpr();

		if (obsExpr.size() == 1) {
			ValueSpecification duration = (ValueSpecification) obsExpr.get(0);
			VSLTreeNode node = VSLTreeNode.create(duration);
			childs.add(node);
		} else if (obsExpr.size() == 2) {
			label += label + " " + "(instant - Instant)";
			ValueSpecification instant1 = (ValueSpecification) obsExpr.get(0);
			ValueSpecification instant2 = (ValueSpecification) obsExpr.get(0);

			VSLTreeNode node = VSLTreeNode.create(instant1);
			childs.add(node);

			node = VSLTreeNode.create(instant2);
			childs.add(node);
		}

	}

	public VSLTreeNode(JitterExpression vsl) {
		element = vsl;
		label = "Jitter Expression";

		childs = new ArrayList<VSLTreeNode>();
		EList obsExpr = vsl.getObsExpr();

		if (obsExpr.size() == 1) {
			ValueSpecification duration = (ValueSpecification) obsExpr.get(0);
			VSLTreeNode node = VSLTreeNode.create(duration);
			childs.add(node);
		} else if (obsExpr.size() == 2) {
			label += label + " " + "(instant, Instant)";
			ValueSpecification instant1 = (ValueSpecification) obsExpr.get(0);
			ValueSpecification instant2 = (ValueSpecification) obsExpr.get(0);

			VSLTreeNode node = VSLTreeNode.create(instant1);
			childs.add(node);

			node = VSLTreeNode.create(instant2);
			childs.add(node);
		}

	}

	public VSLTreeNode(ObsCallExpression vsl) {
		element = vsl;
		childs = new ArrayList<VSLTreeNode>();

		label = "Observation Call Expression";

		VSLTreeNode node = null;

		Observation observation = vsl.getObservation();
		node = new VSLTreeNode(observation);
		childs.add(node);

		ValueSpecification occurIndexExpr = vsl.getOccurIndexExpr();
		if (occurIndexExpr != null) {
			node = VSLTreeNode.create(occurIndexExpr);
			node.setLabel("occur index=" + node.getLabel());
			childs.add(node);
		}

		ValueSpecification conditionExpr = vsl.getConditionExpr();
		if (conditionExpr != null) {
			node = VSLTreeNode.create(conditionExpr);
			node.setLabel("when " + node.getLabel());
			childs.add(node);
		}

	}

	// general
	public static VSLTreeNode create(ValueSpecification vsl) {
		if (vsl instanceof LiteralBoolean)
			return new VSLTreeNode((LiteralBoolean) vsl);
		if (vsl instanceof LiteralReal)
			return new VSLTreeNode((LiteralReal) vsl);
		else if (vsl instanceof LiteralString)
			return new VSLTreeNode((LiteralString) vsl);
		else if (vsl instanceof LiteralInteger)
			return new VSLTreeNode((LiteralInteger) vsl);
		else if (vsl instanceof LiteralUnlimitedNatural)
			return new VSLTreeNode((LiteralUnlimitedNatural) vsl);
		else if (vsl instanceof LiteralNull)
			return new VSLTreeNode((LiteralNull) vsl);
		else if (vsl instanceof LiteralDateTime)
			return new VSLTreeNode((LiteralDateTime) vsl);
		else if (vsl instanceof LiteralDefault)
			return new VSLTreeNode((LiteralDefault) vsl);
		else if (vsl instanceof EnumerationSpecification)
			return new VSLTreeNode((EnumerationSpecification) vsl);
		else if (vsl instanceof PropertyCallExpression)
			return new VSLTreeNode((PropertyCallExpression) vsl);
		else if (vsl instanceof VariableCallExpression)
			return new VSLTreeNode((VariableCallExpression) vsl);
		else if (vsl instanceof Variable)
			return new VSLTreeNode((Variable) vsl);
		else if (vsl instanceof OperationCallExpression)
			return new VSLTreeNode((OperationCallExpression) vsl);
		else if (vsl instanceof ConditionalExpression)
			return new VSLTreeNode((ConditionalExpression) vsl);
		else if (vsl instanceof IntervalSpecification)
			return new VSLTreeNode((IntervalSpecification) vsl);
		else if (vsl instanceof CollectionSpecification)
			return new VSLTreeNode((CollectionSpecification) vsl);
		else if (vsl instanceof TupleSpecification)
			return new VSLTreeNode((TupleSpecification) vsl);
		else if (vsl instanceof ChoiceSpecification)
			return new VSLTreeNode((ChoiceSpecification) vsl);
		else if (vsl instanceof InstantExpression)
			return new VSLTreeNode((InstantExpression) vsl);
		else if (vsl instanceof JitterExpression)
			return new VSLTreeNode((JitterExpression) vsl);
		else if (vsl instanceof DurationExpression)
			return new VSLTreeNode((DurationExpression) vsl);
		else if (vsl instanceof ObsCallExpression)
			return new VSLTreeNode((ObsCallExpression) vsl);
		else
			return null;
	}

	protected VSLTreeNode(VSLTreeNode inner) {
		label = "value specification";
		childs = new ArrayList<VSLTreeNode>();
		childs.add(inner);
	}

	public static VSLTreeNode makeRoot(ValueSpecification vsl) {
		VSLTreeNode node = VSLTreeNode.create(vsl);
		return new VSLTreeNode(node);
	}
}
