/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: VSLDate.java,v 1.4 2007/10/24 07:07:28 fnizou Exp $
 */



package com.thalesgroup.marte.vsl;

import java.text.DecimalFormat;

/**
 * This class represent a Data matching VSL DateTime.
 * @author T0081227 Francois NIZOU july 6 1007
 */
public class VSLDate {

	private static final String MON = "Mon";

	private static final String SUN = "Sun";

	private static final String SAT = "Sat";

	private static final String FRY = "Fry";

	private static final String THU = "Thu";

	private static final String WED = "Wed";

	private static final String TUE = "Tue";

	public Integer year;

	public Integer month;

	public Integer dayOfMonth;

	public DayOfWeek dayOfWeek = null;

	public Integer hour;

	public Integer min;

	public Integer sec;

	public Integer centisec;

	/**
	 * @return the date at VSL concrete format for DataTime
	 */ 
	@Override
	public String toString() {
		String str = "";
		String date = "";
		String day = "";
		String time = "";

		DecimalFormat twoDigitFormat = new DecimalFormat("00");
		DecimalFormat fourDigitFormat = new DecimalFormat("0000");

		if (year != null)
			date += fourDigitFormat.format(year) + "/"
					+ twoDigitFormat.format(month) + "/"
					+ twoDigitFormat.format(dayOfMonth);

		if (hour != null) {
			time += twoDigitFormat.format(hour) + ":"
					+ twoDigitFormat.format(min) + ":"
					+ twoDigitFormat.format(sec);
			if (centisec != null)
				time += ":" + twoDigitFormat.format(centisec);

		}

		if (dayOfWeek != null)
			day = dayOfWeek.toString();

		str += time;

		if (date != "") {
			if (str.length() > 0)
				str += " ";
			str += date;
		}

		if (day != "") {
			if (str.length() > 0)
				str += " ";
			str += day;
		}

		str = "#" + str + "#";
		return str;
	};

	/**
	 * the day of the week
	 * 
	 */
	public enum DayOfWeek {
		MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY;

		public String toString() {
			switch (this) {
			case MONDAY:
				return MON;
			case TUESDAY:
				return TUE;
			case WEDNESDAY:
				return WED;
			case THURSDAY:
				return THU;
			case FRIDAY:
				return FRY;
			case SATURDAY:
				return SAT;
			case SUNDAY:
				return SUN;
			default:
				return "";
			}
		}
	}

	/**
	 * this method fills the fields hour, min, sec.
	 * @param time
	 *            a string matching VSL concrete syntaxe for time
	 * @throws Exception if format or numbers are incorrect.
	 */
	public void parseTime(String time) throws Exception {
		String[] units = time.split(":");

		int h = Integer.parseInt(units[0]);
		if (h > 23)
			throw new Exception("hour must be between 00 and 23");

		int m = Integer.parseInt(units[1]);
		if (m > 59)
			throw new Exception("minute must be between 00 and 59");

		int s = Integer.parseInt(units[2]);
		if (m > 59)
			throw new Exception("secondes must be between 00 and 59");

		this.hour = h;
		this.min = m;
		this.sec = s;

		if (units.length == 4) {
			int cs = Integer.parseInt(units[3]);
			this.centisec = cs;
		}
	}

	/**
	 * This method fills the fields year, month, dayOfMonth
	 * @param date
	 *            a string matching VSL concrete syntaxe for date
	 * @throws Exception
	 */
	public void parseDate(String date) throws Exception {
		String[] units = date.split("/");

		int y = Integer.parseInt(units[0]);

		int m = Integer.parseInt(units[1]);
		if (m > 12 | m == 0)
			throw new Exception(m
					+ " is not a valid month. Month is between 01 and 12");

		int d = Integer.parseInt(units[2]);
		if (d > 31 | d == 0)
			throw new Exception(d
					+ " is not a valid day. day is between 01 and 31");

		this.year = y;
		this.month = m;
		this.dayOfMonth = d;
	}

	/**
	 * this method fills the field dayOfWeek
	 * @param day
	 *            a string matching VSL concrete syntaxe for day of week. <br/>
	 *            see {@link DayOfWeek} for more details.
	 * @throws Exception
	 */
	public void parseDayOfWeek(String day) throws Exception {
		if (day.equals(MON))
			this.dayOfWeek = DayOfWeek.MONDAY;
		else if (day.equals(TUE))
			this.dayOfWeek = DayOfWeek.TUESDAY;
		else if (day.equals(THU))
			this.dayOfWeek = DayOfWeek.THURSDAY;
		else if (day.equals(WED))
			this.dayOfWeek = DayOfWeek.WEDNESDAY;
		else if (day.equals(SAT))
			this.dayOfWeek = DayOfWeek.SATURDAY;
		else if (day.equals(SUN))
			this.dayOfWeek = DayOfWeek.SUNDAY;
		else
			throw new Exception("Incorrect day");
	}
}
