/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: IModelFacade.java,v 1.4 2007/10/15 09:30:05 fnizou Exp $
 */
package com.cea.nfp.parsers.modelgenerator;

import java.util.ArrayList;

import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Observation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.TypedElement;

import VSL.Variable;

/**
 * This Interface Implements the pattern Facade. This facade hide model Access
 * (or hide it the best we can). Althought the editor still use
 * org.eclipse.uml2.uml API, the navigation is mostly done here, hence it does
 * reduce the numbers of elements to implements when using with a different
 * model. Any integration of the Editor for a non org.eclipse.uml2.uml model
 * should implements this facade.
 * 
 * A standart implementation for org.eclipse.uml2.uml is {@link UMLModelFacade}
 * 
 * @author T0081227 Francois NIZOU - 11 juil. 07
 * 
 */
public interface IModelFacade {

	/**
	 * Note: operation Called on Datatype are:
	 * <ul>
	 * <li>{@link DataType#getOwnedAttributes()}</li>
	 * <li>{@link Element#getAppliedStereotypes()}</li>
	 * <li>{@link NamedElement#getName()}</li>
	 * </ul>
	 * 
	 * @return all the {@link DataType} of the model and its dependencies.
	 */
	ArrayList<DataType> getDataTypes();

	/**
	 * 
	 * @param startwith
	 * @return all Datatypes from model and its dependencies where name start
	 *         with startwith
	 */
	ArrayList<DataType> getDataTypesStartWith(String startwith);

	/**
	 * 
	 * @param startwith
	 * @return all Datatypes from model and its dependencies where name equal
	 *         name
	 */
	ArrayList<DataType> getDataTypesByName(String name);

	/**
	 * 
	 * @return All the VSL variables of the Model.
	 */
	ArrayList<Variable> getVariables();

	/**
	 * 
	 * @param prefix
	 * @return all the VSL variables of the Model Starting with startwith
	 */
	ArrayList<Variable> getVariablesStartWith(String prefix);

	/**
	 * 
	 * @param namespace
	 * @param name
	 * @return all Variables called namespace.name
	 */
	ArrayList<Variable> getVariablesByName(String namespace, String name);

	/**
	 * Note: operation Called on Property are:
	 * <ul>
	 * <li>{@link TypedElement#getType()}</li>
	 * <li>{@link NamedElement#getName()}</li>
	 * </ul>
	 * 
	 * @return all the properties of the model
	 */
	ArrayList<Property> getProperties();

	/**
	 * 
	 * @param startwith
	 * @return all the properties from the model which name start by startwith
	 */
	ArrayList<Property> getPropertiesStartWith(String startwith);

	/**
	 * 
	 * @param namespace
	 * @param name
	 * @return all propertyname called namespace.name (FQN)
	 */
	ArrayList<Property> getpropertiesByName(String namespace, String name);

	/**
	 * 
	 * @param typedElement
	 * @return Return the type of any Typed Element
	 */
	DataType typeof(TypedElement typedElement);

	/**
	 * 
	 * @param property
	 * @return true is the property is within the current ExpressionContext.
	 */
	boolean isLocalProperty(Property property);

	/**
	 * 
	 * @param namedElement
	 * @return the FQN (with dot, not semicolon) of an element, considering
	 *         ExpressionContext
	 */
	String getQualifiedName(NamedElement namedElement);

	/**
	 * Note: operation Called on EnumerationLiteral are:
	 * <ul>
	 * <li>{@link EnumerationLiteral#getEnumeration()}</li>
	 * <li>{@link NamedElement#getName()}</li>
	 * </ul>
	 * 
	 * @param name
	 *            the name of the EnumerationLiteral
	 * @param isStartWith
	 *            set false if the EnumerationLiteral as a name strictly equal.
	 * @return all the EnumerationLiteral mathing name.
	 */
	ArrayList<EnumerationLiteral> getEnumerationsByName(String name,
			boolean isStartWith);

	/**
	 * Find a collection of Datatype for TupleType resolution.
	 * 
	 * @param tupleItemNames
	 *            the Attribute names
	 * @return return all Datatype, stereotyped by TupleType which attributes
	 *         match tupleItemNames
	 */
	ArrayList<DataType> findTupleByItemNames(ArrayList<String> tupleItemNames);

	/**
	 * Find a collection of Datatype for ChoiceType resolution.
	 * 
	 * @param chosenAlternative
	 * @param valueType
	 *            the dataType associed with the chosen alternative.
	 * @return all Datatype, stereotyped by ChoiceType, mathing the parameters.
	 */
	DataType getChoice(String chosenAlternative, DataType valueType);

	/**
	 * 
	 * 
	 * @return the fully Qualified name of the namespace of the current element
	 */
	String getCurrentContextName();

	/**
	 * Get the context Name from the current element (this element should be the
	 * one owning the VSLExpression).  The ctx is computed according to context name,
	 * i.e the fqn of the first UML element stereotyped by ExpressionContext containing
	 * the current element.
	 */
	String getCurrentVariableContextName();

	/**
	 * 
	 * @param name
	 * @return all Observation from the model which name are name.
	 */
	ArrayList<Observation> getObservationsByName(String name);

	/**
	 * 
	 * @param prefix
	 * @return all Observation from the model which name startwith prefix.
	 */
	ArrayList<Observation> getObservationsStartWith(String prefix);

	/**
	 * Note: operation Called on Observation are:
	 * <ul>
	 * <li>{@link NamedElement#getName()}</li>
	 * <li> and instanceof comparaison for TimeObservation or
	 * DurationObservation {@link Observation}</li>
	 * </ul>
	 * 
	 * @return all the Observation of the Model
	 */
	ArrayList<Observation> getObservations();
}
