/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: Linker.java,v 1.11 2007/10/24 07:05:26 fnizou Exp $
 */
package com.cea.nfp.parsers.modelgenerator;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.DurationObservation;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralNull;
import org.eclipse.uml2.uml.LiteralSpecification;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Observation;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.StringExpression;
import org.eclipse.uml2.uml.TimeObservation;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.TypedElement;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;

import VSL.ChoiceSpecification;
import VSL.CollectionSpecification;
import VSL.ConditionalExpression;
import VSL.DurationExpression;
import VSL.EnumerationSpecification;
import VSL.InstantExpression;
import VSL.IntervalSpecification;
import VSL.JitterExpression;
import VSL.LiteralDateTime;
import VSL.LiteralDefault;
import VSL.LiteralReal;
import VSL.ObsCallExpression;
import VSL.OperationCallExpression;
import VSL.PropertyCallExpression;
import VSL.TimeExpression;
import VSL.TupleItemValue;
import VSL.TupleSpecification;
import VSL.VSLFactory;
import VSL.Variable;
import VSL.VariableCallExpression;

import com.cea.nfp.parsers.texteditor.vsldatatypes.IContext;
import com.cea.nfp.parsers.texteditor.vsldatatypes.VSLComplexTypeUtil;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.PrimitivesTypes;

/**
 * This class complete and type check a VSL model built by the parser.
 * 
 * @author T0081227 Francois NIZOU - 7 aot 07
 * 
 */
public class Linker {

	protected IModelFacade facade;

	protected IDecideStrategy strategy;

	/**
	 * 
	 * @param facade
	 * @param strategy
	 */
	public Linker(IModelFacade facade, IDecideStrategy strategy) {
		this.facade = facade;
		this.strategy = strategy;
	}

	/**
	 * Link and typeCheck the VSL model VSL.<br/> The Expected type might be
	 * used to type check or help on the VSL Expression name and type
	 * resolution. This might be null but might lower the chance of a correct
	 * linkage. The strategy used is the one from the constructor.
	 * 
	 * @param vsl
	 * @param expected
	 * @return
	 * @throws TypeOrLinkException
	 */
	public ValueSpecification link(ValueSpecification vsl, DataType expected)
			throws TypeOrLinkException {
		return link(vsl, expected, this.strategy);
	}

	/**
	 * this{@link #link(ValueSpecification, DataType)}
	 * 
	 * @param vsl
	 * @param expected
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	public ValueSpecification link(ValueSpecification vsl, DataType expected,
			IDecideStrategy strategy) throws TypeOrLinkException {
		if (expected != null && expected.getName().equals("VSL_Expression"))
			expected = null;

		if (vsl instanceof StringExpression) {
			StringExpression se = (StringExpression) vsl;
			return resolveIdentifier(se.getName(), expected, strategy);
		} else if (vsl instanceof Variable) {
			return linkVariable((Variable) vsl, expected);
		} else if (vsl instanceof OperationCallExpression) {
			return linkCallOperation((OperationCallExpression) vsl, expected);
		} else if (vsl instanceof LiteralSpecification) {
			if (expected != null) {

				// Special case for null
				if (vsl instanceof LiteralNull || vsl instanceof LiteralDefault)
					return vsl;

				DataType type = Typer.type(vsl, facade).get(0);
				String name = type.getName();

				// special case for Unlimited Natural & Integer
				if (expected.getName().equals(PrimitivesTypes.UNLIMITEDNATURAL)
						&& name.equals(PrimitivesTypes.INTEGER)) {
					LiteralInteger li = (LiteralInteger) vsl;
					if (li.getValue() > 0) {
						LiteralUnlimitedNatural lun = UMLFactory.eINSTANCE
								.createLiteralUnlimitedNatural();
						lun.setValue(li.getValue());
						return lun;
					} else {
						throw new TypeOrLinkException(
								"Unlimited Natural can only be  positive or null");
					}
				}

				if (!Typer.isTypeOf(vsl, expected, facade))
					throw new BadTypeException(type, expected);
			}

			return copyOf((LiteralSpecification) vsl);
		} else if (vsl instanceof TupleSpecification) {
			TupleSpecification ts = (TupleSpecification) vsl;

			// in some case, parser recognizes time expr as tuple
			if (ts.getTupleItem().size() == 1) {
				TupleItemValue item = (TupleItemValue) ts.getTupleItem().get(0);
				if (item.getTupleItemName() == null
						|| item.getTupleItemName().length() == 0) {
					ValueSpecification itemValue = item.getItemValue();
					if (itemValue instanceof OperationCallExpression) {
						OperationCallExpression oce = (OperationCallExpression) itemValue;
						if (oce.getOperation().equals("-")) {
							DurationExpression ie = transformTuple2Duration(oce
									.getArgument());

							try {
								return link(ie, expected);
							} catch (Exception e) {

							}
						} else if (oce.getOperation().equals("+")) {
							InstantExpression ie = transformTuple2Instant(oce
									.getArgument());

							try {
								return link(ie, expected);
							} catch (Exception e) {

							}
						}
					}
				}
			}

			return linkTupleSpecification((TupleSpecification) vsl, expected,
					strategy);
		} else if (vsl instanceof ChoiceSpecification) {
			return linkChoiceSpecification((ChoiceSpecification) vsl, expected,
					strategy);
		} else if (vsl instanceof ConditionalExpression) {
			ConditionalExpression cond = (ConditionalExpression) vsl;
			ConditionalExpression newCond = VSLFactory.eINSTANCE
					.createConditionalExpression();
			ValueSpecification ifFalseExpr = link(cond.getIfFalseExpr(),
					expected, strategy);
			ValueSpecification ifTrueExpr = link(cond.getIfTrueExpr(),
					expected, strategy);
			DataType bool = facade.getDataTypesByName(PrimitivesTypes.BOOLEAN)
					.get(0);
			ValueSpecification conditionExpr = link(cond.getConditionExpr(),
					bool, strategy);
			newCond.setConditionExpr(conditionExpr);
			newCond.setIfFalseExpr(ifFalseExpr);
			newCond.setIfTrueExpr(ifTrueExpr);
			return newCond;
		} else if (vsl instanceof InstantExpression) {
			ValueSpecification ie = linkInstantExpression(
					(InstantExpression) vsl, strategy);
			if (expected != null) {
				// DataType type = Typer.type(ie, facade).get(0);

				if (!Typer.isTypeOf(ie, expected, facade))
					throw new BadTypeException(Typer.type(ie, facade).get(0),
							expected);
			}
			return ie;
		} else if (vsl instanceof JitterExpression) {
			ValueSpecification je = linkJitterExpression(
					(JitterExpression) vsl, strategy);
			if (expected != null) {
				DataType type = Typer.type(je, facade).get(0);
				if (!Typer.isTypeOf(je, expected, facade))
					throw new BadTypeException(type, expected);
			}
			return je;
		} else if (vsl instanceof DurationExpression) {
			ValueSpecification de = linkDurationExpression(
					(DurationExpression) vsl, strategy);
			if (expected != null) {
				DataType type = Typer.type(de, facade).get(0);
				if (!Typer.isTypeOf(de, expected, facade))
					throw new BadTypeException(type, expected);
			}
			return de;
		} else if (vsl instanceof TimeExpression) {
			ValueSpecification te = linkTimeExpression((TimeExpression) vsl,
					strategy);
			DataType type = Typer.type(te, facade).get(0);
			if (expected != null) {
				if (!Typer.isTypeOf(te, expected, facade))
					throw new BadTypeException(type, expected);
			}
			return te;
		} else if (vsl instanceof ObsCallExpression) {
			ObsCallExpression obsCall = linkObsCallExpression(
					(ObsCallExpression) vsl, null, strategy);
			DataType type = Typer.type(obsCall, facade).get(0);
			if (expected != null) {
				if (!Typer.isTypeOf(vsl, expected, facade))
					throw new BadTypeException(type, expected);
			}
			return obsCall;
		} else if (vsl instanceof IntervalSpecification) {
			ValueSpecification vs = linkInterval((IntervalSpecification) vsl,
					expected, strategy);
			return vs;
		} else if (vsl instanceof CollectionSpecification) {
			ValueSpecification vs = linkCollection(
					(CollectionSpecification) vsl, expected, strategy);
			return vs;
		} else {
			return vsl;
		}
	}

	private DurationExpression transformTuple2Duration(EList argument) throws TypeOrLinkException {
		Object[] args = argument.toArray();
		ValueSpecification vs1 = (ValueSpecification) args[0];

		DurationExpression result = VSLFactory.eINSTANCE
				.createDurationExpression();

		ValueSpecification leftObs = null;

		if (vs1 instanceof StringExpression) {
			leftObs = VSLFactory.eINSTANCE.createObsCallExpression();
			leftObs.setNameExpression((StringExpression) vs1);
		} else {
			TimeExpression te = (TimeExpression) vs1;
			leftObs = (ValueSpecification) te.getObsExpr().get(0);
		}

		ValueSpecification vs2 = (ValueSpecification) args[1];

		ValueSpecification rightObs = null;

		if (vs2 instanceof StringExpression) {
			rightObs = VSLFactory.eINSTANCE.createObsCallExpression();
			rightObs.setNameExpression((StringExpression) vs2);
		} else {
			TimeExpression te = (TimeExpression) vs2;
			rightObs = (ValueSpecification) te.getObsExpr().get(0);
		}

		result.getObsExpr().add(leftObs);
		result.getObsExpr().add(rightObs);

		return result;
	}

	private InstantExpression transformTuple2Instant(EList argument)
			throws TypeOrLinkException {
		Object[] args = argument.toArray();
		ValueSpecification vs1 = (ValueSpecification) args[0];

		InstantExpression result = VSLFactory.eINSTANCE
				.createInstantExpression();

		ValueSpecification leftObs = null;

		if (vs1 instanceof StringExpression) {
			leftObs = VSLFactory.eINSTANCE.createObsCallExpression();
			leftObs.setNameExpression((StringExpression) vs1);
		} else {
			TimeExpression te = (TimeExpression) vs1;
			leftObs = (ValueSpecification) te.getObsExpr().get(0);
		}

		ValueSpecification vs2 = (ValueSpecification) args[1];

		ValueSpecification rightObs = null;

		if (vs2 instanceof StringExpression) {
			rightObs = VSLFactory.eINSTANCE.createObsCallExpression();
			rightObs.setNameExpression((StringExpression) vs2);
		} else {
			TimeExpression te = (TimeExpression) vs2;
			rightObs = (ValueSpecification) te.getObsExpr().get(0);
		}

		result.getObsExpr().add(leftObs);
		result.getObsExpr().add(rightObs);

		return result;
	}

	private ValueSpecification linkCollection(
			CollectionSpecification specification, DataType expected,
			IDecideStrategy strategy) throws TypeOrLinkException {
		CollectionSpecification cs = VSLFactory.eINSTANCE
				.createCollectionSpecification();

		DataType colType = null;

		if (expected != null) {

			String compositeType = VSLComplexTypeUtil
					.getCompositeType(expected);
			if (compositeType == null
					|| (!compositeType.equals(VSL.STEREOTYPE_COLLECTION_TYPE)))
				throw new TypeOrLinkException("expected " + expected.getName()
						+ " found a collection specification");

			Stereotype intervalStereotype = null;
			EList stereotypes = expected.getAppliedStereotypes();
			for (Object object : stereotypes) {
				Stereotype s = (Stereotype) object;
				if (s.getName().equals(VSL.STEREOTYPE_COLLECTION_TYPE)) {
					intervalStereotype = s;
					break;
				}
			}
			Property prop = (Property) expected.getValue(intervalStereotype,
					VSL.VSL_COLLECTION_ATTRIB);

			colType = facade.typeof(prop);
			cs.setType(expected);
		}

		EList itemValues = specification.getItemValue();

		for (Object object : itemValues) {
			ValueSpecification vs = (ValueSpecification) object;
			ValueSpecification linkedVs = link(vs, colType, strategy);
			cs.getItemValue().add(linkedVs);
		}

		return cs;
	}

	private ValueSpecification linkInterval(
			IntervalSpecification specification, DataType expected,
			IDecideStrategy strategy) throws TypeOrLinkException {
		IntervalSpecification result = null;
		if (expected != null) {
			result = VSLFactory.eINSTANCE.createIntervalSpecification();
			String compositeType = VSLComplexTypeUtil
					.getCompositeType(expected);
			if (compositeType == null
					|| (!compositeType.equals(VSL.STEREOTYPE_INTERVAL_TYPE)))
				throw new TypeOrLinkException("expected " + expected.getName()
						+ " found an interval specification");

			org.eclipse.uml2.uml.Stereotype intervalStereotype = null;
			EList stereotypes = expected.getAppliedStereotypes();
			for (Object object : stereotypes) {
				org.eclipse.uml2.uml.Stereotype s = (org.eclipse.uml2.uml.Stereotype) object;
				if (s.getName().equals(VSL.STEREOTYPE_INTERVAL_TYPE)) {
					intervalStereotype = s;
					break;
				}
			}
			Property prop = (Property) expected.getValue(intervalStereotype,
					VSL.VSL_INTERVAL_ATTRIB);

			DataType intervalType = facade.typeof(prop);

			ValueSpecification max = specification.getMax();
			ValueSpecification linkedMax = link(max, intervalType, strategy);
			ValueSpecification min = specification.getMin();
			ValueSpecification linkedMin = link(min, intervalType, strategy);

			result.setMax(linkedMax);
			result.setMin(linkedMin);
			result.setType(expected);

		} else {
			result = VSLFactory.eINSTANCE.createIntervalSpecification();
			ValueSpecification max = specification.getMax();
			ValueSpecification linkedMax = link(max, null, strategy);
			ValueSpecification min = specification.getMin();
			ValueSpecification linkedMin = link(min, null, strategy);
			result.setMax(linkedMax);
			result.setMin(linkedMin);
		}

		result.setIsLowerOpen(specification.isIsLowerOpen());
		result.setIsUpperOpen(specification.isIsUpperOpen());
		return result;
	}

	/**
	 * 
	 * @param choice
	 * @param expected
	 * @param strategy2
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	protected ValueSpecification linkChoiceSpecification(
			ChoiceSpecification choice, DataType expected,
			IDecideStrategy strategy) throws TypeOrLinkException {

		ChoiceSpecification newChoice = VSLFactory.eINSTANCE
				.createChoiceSpecification();
		newChoice.setChosenAlternative(choice.getChosenAlternative());

		ValueSpecification vsl = choice.getValue();

		ValueSpecification linkedVsl = null;
		try {
			linkedVsl = this.link(vsl, null, strategy);
		} catch (Exception e) {
			if (vsl instanceof TupleSpecification) {
				TupleSpecification ts = (TupleSpecification) vsl;
				if (ts.getTupleItem().size() == 1) {
					TupleItemValue tiv = (TupleItemValue) ts.getTupleItem()
							.get(0);
					linkedVsl = this.link(tiv.getItemValue(), null, strategy);
				}
			}
		}

		newChoice.setValue(linkedVsl);
		DataType choiceType = Typer.type(newChoice, facade).get(0);

		if (choiceType == null)
			throw new UnresolvedNameException(
					"the choice corresponding to the chosen alternative "
							+ choice.getChosenAlternative());

		EList attributes = choiceType.getAllAttributes();
		for (Object object : attributes) {
			Property property = (Property) object;
			if (property.getName().equals(choice.getChosenAlternative())) {
				newChoice.setChoiceAttribute(property);
				break;
			}
		}

		newChoice.setType(choiceType);
		if (expected != null && !Typer.isTypeOf(newChoice, expected, facade))
			throw new BadTypeException(choice.getChosenAlternative(), expected);

		return newChoice;
	}

	/**
	 * 
	 * @param specification
	 * @param expected
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */

	protected ValueSpecification linkTupleSpecification(
			TupleSpecification tuple, DataType expected,
			IDecideStrategy strategy) throws TypeOrLinkException {

		TupleLinker tupleLinker = new TupleLinker(facade, this);
		return tupleLinker.linkTupleSpecification(tuple, expected, strategy);
	}

	/**
	 * 
	 * @param variable
	 * @param expected
	 * @return
	 * @throws UnresolvedNameException
	 * @throws BadTypeException
	 */
	protected ValueSpecification linkVariable(Variable variable,
			DataType expected) throws TypeOrLinkException {
		Variable newVar = VSLFactory.eINSTANCE.createVariable();
		newVar.setName(variable.getName());
		String dataTypeName = variable.getDataTypeName();
		ArrayList<DataType> datatypes = facade.getDataTypesByName(dataTypeName);
		if (datatypes == null || datatypes.size() == 0)
			throw new UnresolvedNameException(dataTypeName);
		DataType dataType = datatypes.get(0);
		newVar.setDatatype(dataType);
		newVar.setDataTypeName(dataType.getName());
		newVar.setType(dataType);

		ValueSpecification initExpression = variable.getInitExpression();
		if (initExpression != null) {
			ValueSpecification linkedInitExpr = link(initExpression, dataType,
					strategy);
			newVar.setInitExpression(linkedInitExpr);
		}

		if (expected == null)
			return newVar;

		if (expected.getName().equals(dataTypeName)) {
			return newVar;
		} else {
			throw new BadTypeException(variable.getName(), expected);
		}

	}

	/**
	 * 
	 * @param name
	 * @param expected
	 * @param strategy
	 * @return
	 * @throws UnresolvedNameException
	 * @throws BadTypeException
	 */
	protected ValueSpecification resolveIdentifier(String name,
			DataType expected, IDecideStrategy strategy)
			throws TypeOrLinkException {

		ArrayList<Object> possibleHit = new ArrayList<Object>();
		int dotIndex = name.lastIndexOf(".");
		String id = "";
		String ns = "";

		if (dotIndex == -1) { // no namespace, can be enums, property or
			// variable
			if (expected != null && expected instanceof Enumeration) {
				Enumeration e = (Enumeration) expected;
				EnumerationLiteral ownedLiteral = e.getOwnedLiteral(name);

				if (ownedLiteral != null) { // this is an
					// enumerationSpecification
					EnumerationSpecification eSpecif = VSLFactory.eINSTANCE
							.createEnumerationSpecification();
					eSpecif.setEnumLiteral(ownedLiteral);
					eSpecif.setName(ownedLiteral.getName());
					eSpecif.setType(e);
					return eSpecif;
				}
			} else {
				possibleHit.addAll(facade.getEnumerationsByName(name, false));
				possibleHit.addAll(facade.getObservationsByName(name));
			}
			id = name;
			ns = "";
		} else {
			ns = name.substring(0, dotIndex);
			id = name.substring(dotIndex + 1, name.length());
		}

		possibleHit.addAll(findVarOrPropsByName(ns, id, expected));
		if (possibleHit != null && possibleHit.size() > 0) {
			Object result = null;
			if (possibleHit.size() == 1) {
				result = possibleHit.get(0);

				if (!(result instanceof Observation)
						&& result instanceof TypedElement) {
					TypedElement typedElement = (TypedElement) result;

					DataType typeof = facade.typeof(typedElement);
					if (typeof == null) {
						throw new TypeOrLinkException("found a reference to '"
								+ name
								+ "' but it was not of a correct Datatype");
					}
					if (expected != null
							&& !(typeof.getName().equals(expected.getName())))
						throw new BadTypeException(name, expected);
				}
			} else {
				ArrayList<Object> filteredPossibleHit = new ArrayList<Object>();
				if (expected != null) {
					for (Object object : possibleHit) {
						if (object instanceof Observation) {
							filteredPossibleHit.add(object);
						} else if (object instanceof EnumerationLiteral) {
							continue;
						} else {
							TypedElement te = (TypedElement) object;
							if (facade.typeof(te).getName().equals(
									expected.getName()))
								filteredPossibleHit.add(te);
						}

					}
				} else {
					filteredPossibleHit = possibleHit;
				}
				if (filteredPossibleHit.size() > 0) {
					result = strategy.decide(filteredPossibleHit,
							IContext.NAME_EXPRESSION);
				} else {
					throw new BadTypeException(name, expected);
				}
			}

			if (result instanceof Variable) {
				VariableCallExpression ce = VSLFactory.eINSTANCE
						.createVariableCallExpression();
				ce.setDefiningVariable((Variable) result);
				ce.setVariable(name);
				ce.setType(facade.typeof((Variable) result));
				return ce;
			} else if (result instanceof Property) {
				PropertyCallExpression pce = VSLFactory.eINSTANCE
						.createPropertyCallExpression();
				pce.setDefiningProperty((Property) result);
				pce.setType(facade.typeof((Property) result));
				return pce;
			} else if (result instanceof Observation) {
				TimeExpression te = null;
				if (result instanceof TimeObservation)
					te = VSLFactory.eINSTANCE.createInstantExpression();
				else
					te = VSLFactory.eINSTANCE.createDurationExpression();
				ObsCallExpression oce = VSLFactory.eINSTANCE
						.createObsCallExpression();
				oce.setObservation((Observation) result);
				oce.setType(Typer.type(oce, facade).get(0));
				te.getObsExpr().add(oce);

				DataType type = Typer.type(te, facade).get(0);
				if (expected != null && !Typer.isTypeOf(te, expected, facade))
					throw new BadTypeException(type, expected);

				return te;
			} else if (result instanceof EnumerationLiteral) {
				EnumerationLiteral eliteral = (EnumerationLiteral) result;
				EnumerationSpecification eSpecif = VSLFactory.eINSTANCE
						.createEnumerationSpecification();
				eSpecif.setEnumLiteral(eliteral);
				eSpecif.setName(eliteral.getName());
				eSpecif.setType(eliteral.getEnumeration());
				return eSpecif;
			}
		} else { // noname found
			ArrayList<Object> possiblesMistyped = findVarOrPropsByName(ns, id,
					expected);
			if (possiblesMistyped == null || possiblesMistyped.size() == 0)
				throw new UnresolvedNameException(name);
			else {
				Object o = possiblesMistyped.get(0);
				name = "";
				String typeName = "";
				if (o instanceof NamedElement) {
					name = ((org.eclipse.uml2.uml.NamedElement) o).getName();
				}
				if (o instanceof TypedElement) {
					typeName = ((TypedElement) o).getType().getName();
				}

				throw new TypeOrLinkException("Excepted type "
						+ expected.getName() + " found " + name + " of type "
						+ typeName);
			}
		}

		return null;
	}

	/**
	 * @param name
	 */
	protected ArrayList<Object> findVarOrPropsByName(String ns, String name,
			DataType expected) {
		ArrayList<Object> possibleHit = new ArrayList<Object>();
		ArrayList<Property> matchingProperties = facade.getpropertiesByName(ns,
				name);

		ArrayList<Variable> matchingVariables = facade.getVariablesByName(ns,
				name);
		ArrayList<Variable> linkedMatchingVariables = new ArrayList<Variable>();

		for (Variable variable : matchingVariables) {
			try {
				Variable link = (Variable) this.link(variable, expected);
				linkedMatchingVariables.add(link);
			} catch (Exception e) {

			}
		}

		if (matchingProperties != null && matchingProperties.size() > 0)
			possibleHit.addAll(matchingProperties);
		if (matchingVariables != null && matchingVariables.size() > 0)
			possibleHit.addAll(matchingVariables);

		return possibleHit;
	}

	/**
	 * 
	 * @param expression
	 * @param expected
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	protected ValueSpecification linkCallOperation(
			OperationCallExpression expression, DataType expected)
			throws TypeOrLinkException {
		OperationCallExpression result = VSLFactory.eINSTANCE
				.createOperationCallExpression();
		// First, we resolve name.

		// the object owner of the operation is the first argument
		ValueSpecification object = (ValueSpecification) expression
				.getArgument().get(0);
		expression.getArgument().remove(0);
		ValueSpecification linkedObject = this.link(object, null);

		DataType typeOf = Typer.type(linkedObject, facade).get(0);

		// resolv the possible Operations

		ArrayList<Operation> matchingOperations = new ArrayList<Operation>();
		EList operations = typeOf.getOperations();
		for (Object o : operations) {
			Operation operation = (Operation) o;
			if (operation.getName().equals(expression.getOperation()))
				matchingOperations.add(operation);
		}

		// Then we resolve arguments

		ArrayList<ValueSpecification> LinkedArgs = new ArrayList<ValueSpecification>();
		EList args = expression.getArgument();

		for (Object o : args) {
			LinkedArgs.add(this.link((ValueSpecification) o, null));
		}

		ArrayList<DataType> argsType = new ArrayList<DataType>();
		for (ValueSpecification vsl : LinkedArgs) {
			argsType.add(Typer.type(vsl, facade).get(0));
		}

		// finaly we decide if both match and match with expected type.
		ArrayList<Operation> checkedOperation = new ArrayList<Operation>();
		for (Operation operation : matchingOperations) {
			EList ownedParameters = operation.getOwnedParameters();
			// if numbers of args does not match, this is not the good one

			ArrayList<Parameter> parameters = new ArrayList<Parameter>();
			for (Object o : ownedParameters) {
				Parameter p = (Parameter) o;
				if (!p.getDirection().equals(
						ParameterDirectionKind.RETURN_LITERAL))
					parameters.add(p);
			}

			if (parameters.size() != LinkedArgs.size())
				continue;

			// check if type match
			boolean isOk = true;
			for (int i = 0; i < parameters.size(); i++) {
				Parameter p = parameters.get(i);

				if (!Typer
						.isTypeOf(LinkedArgs.get(i), facade.typeof(p), facade)) {
					isOk = false;
					break;
				}
			}
			if (isOk)
				checkedOperation.add(operation);

		}

		// if we had many hits on operation name resolution, we decide here.
		if (checkedOperation.size() == 0) {
			String operandesStr = "( ";
			for (DataType type : argsType) {
				operandesStr += type.getName() + " ";
			}
			operandesStr += ")";
			throw new UnresolvedNameException("the operation "
					+ typeOf.getName() + "." + expression.getOperation()
					+ operandesStr);
		} else if (checkedOperation.size() == 1) {
			Operation operation = checkedOperation.get(0);
			result.setDefiningOperation(operation);
		} else {
			throw new UnresolvedNameException("ambigous operation "
					+ expression.getOperation());
		}

		if (expected != null) {
			Operation definingOperation = result.getDefiningOperation();
			Parameter returnResult = definingOperation.getReturnResult();
			Type type = definingOperation.getType();
			if (!expected.getName().equals(
					facade.typeof(returnResult).getName()))
				throw new BadTypeException("the operation "
						+ expression.getOperation(), expected);
		}
		// expression.getArgument().clear();
		expression.getArgument().set(0, object);
		result.getArgument().add(0, linkedObject);
		for (ValueSpecification linkedArgs : LinkedArgs) {
			result.getArgument().add(linkedArgs);

		}
		result.setOperation(expression.getOperation());
		return result;
	}

	/**
	 * 
	 * @param ie
	 * @param strategy
	 * @return
	 * @throws UnresolvedNameException
	 * @throws BadTypeException
	 */
	protected ValueSpecification linkInstantExpression(InstantExpression ie,
			IDecideStrategy strategy) throws TypeOrLinkException {
		InstantExpression newInstantExpr = VSLFactory.eINSTANCE
				.createInstantExpression();

		// we re in ( <instant-obs-expr> '+' <duration-obs-expr> ) case

		ObsCallExpression left = (ObsCallExpression) ie.getObsExpr().get(0);
		ObsCallExpression right = (ObsCallExpression) ie.getObsExpr().get(1);

		// first we resolve left and right and check if left is instant and
		// right is duration

		ObsCallExpression linkedLeft = linkObsCallExpression(left,
				UMLPackage.TIME_OBSERVATION, strategy);
		Observation leftObs = linkedLeft.getObservation();
		if (leftObs == null)
			throw new UnresolvedNameException(left.getNameExpression()
					.getName()
					+ " (TimeObservation Expected)");

		ObsCallExpression linkedRight = linkObsCallExpression(right,
				UMLPackage.DURATION_OBSERVATION, strategy);
		Observation rightObs = linkedRight.getObservation();
		if (rightObs == null) // 
			throw new UnresolvedNameException(right.getNameExpression()
					.getName()
					+ "DurationObservation Expected");

		// change the ObsExpr with linked one
		newInstantExpr.getObsExpr().clear();
		newInstantExpr.getObsExpr().add(linkedLeft);
		newInstantExpr.getObsExpr().add(linkedRight);

		return newInstantExpr;
	}

	/**
	 * 
	 * @param obsCall
	 * @param expectedClass
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	protected ObsCallExpression linkObsCallExpression(
			ObsCallExpression obsCall, Integer expectedClass,
			IDecideStrategy strategy) throws TypeOrLinkException {
		// we ought to
		// 1. resolve Observation
		// 2. resolve occurInder (if any)
		// 3. resolve the cond

		ObsCallExpression oce = VSLFactory.eINSTANCE.createObsCallExpression();

		ArrayList<Observation> obss = facade.getObservationsByName(obsCall
				.getNameExpression().getName());

		ArrayList<Observation> filteredObss = null;

		if (expectedClass == null)
			filteredObss = obss;
		else {
			filteredObss = new ArrayList<Observation>();
			for (Observation observation : obss) {
				if (expectedClass == UMLPackage.TIME_OBSERVATION
						&& observation instanceof TimeObservation)
					filteredObss.add(observation);
				if (expectedClass == UMLPackage.DURATION_OBSERVATION
						&& observation instanceof DurationObservation)
					filteredObss.add(observation);
			}
		}

		if (filteredObss == null || filteredObss.size() == 0) {
			throw new UnresolvedNameException("the Observation "
					+ obsCall.getNameExpression().getName());
		}
		if (filteredObss.size() == 1)
			oce.setObservation(filteredObss.get(0));
		else {
			oce.setObservation((Observation) strategy.decide(
					(ArrayList) filteredObss, IContext.TIME_EXPR));
		}

		ValueSpecification occurIndexExpr = obsCall.getOccurIndexExpr();
		if (occurIndexExpr != null) {
			DataType intType = UMLFactory.eINSTANCE.createDataType();
			intType.setName("Integer");
			ValueSpecification linkedOccurIndex = link(occurIndexExpr, intType,
					strategy);
			oce.setOccurIndexExpr(linkedOccurIndex);
		}

		ValueSpecification condExpr = obsCall.getConditionExpr();
		if (condExpr != null) {
			DataType booleanType = UMLFactory.eINSTANCE.createDataType();
			booleanType.setName("Boolean");
			ValueSpecification linkedcondExpr = link(condExpr, booleanType,
					strategy);
			oce.setConditionExpr(linkedcondExpr);
		}

		return oce;
	}

	/**
	 * 
	 * @param timeExpression
	 * @param strategy
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	protected ValueSpecification linkTimeExpression(
			TimeExpression timeExpression, IDecideStrategy strategy)
			throws TypeOrLinkException {
		// we re in case <instant-obs-expr> or <duration-obs-expr>
		// only 1 observer in timeExpression.getObs
		// we have to find out if this is an instant or duration Expression
		ObsCallExpression obsCallExpr = (ObsCallExpression) timeExpression
				.getObsExpr().get(0);

		// we link the obsCallExpr
		ObsCallExpression linkedObsCallExpr = linkObsCallExpression(
				obsCallExpr, null, strategy);
		// we check if this is a TimeObservation or a DurationObservation

		TimeExpression linkedTimeExpression = null;

		if (linkedObsCallExpr.getObservation() instanceof TimeObservation)
			linkedTimeExpression = VSLFactory.eINSTANCE
					.createInstantExpression();
		else
			linkedTimeExpression = VSLFactory.eINSTANCE
					.createDurationExpression();

		linkedTimeExpression.getObsExpr().add(linkedObsCallExpr);
		return linkedTimeExpression;
	}

	/**
	 * 
	 * @param durationExpression
	 * @param strategy
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	private ValueSpecification linkDurationExpression(
			DurationExpression durationExpression, IDecideStrategy strategy)
			throws TypeOrLinkException {

		DurationExpression newDurationExpression = VSLFactory.eINSTANCE
				.createDurationExpression();

		// we re in '(' <instantobs-expr> '-' <instant-obs-expr> ')'case

		ObsCallExpression left = (ObsCallExpression) durationExpression
				.getObsExpr().get(0);
		ObsCallExpression right = (ObsCallExpression) durationExpression
				.getObsExpr().get(1);

		// first we resolve left and right and check if left is instant and
		// right is duration

		ObsCallExpression linkedLeft = linkObsCallExpression(left,
				UMLPackage.TIME_OBSERVATION, strategy);
		Observation leftObs = linkedLeft.getObservation();
		if (leftObs == null)
			throw new UnresolvedNameException(left.getNameExpression()
					.getName()
					+ " (TimeObservation Expected)");

		ObsCallExpression linkedRight = linkObsCallExpression(right,
				UMLPackage.TIME_OBSERVATION, strategy);
		Observation rightObs = linkedRight.getObservation();
		if (rightObs == null) // 
			throw new UnresolvedNameException(right.getNameExpression()
					.getName()
					+ "TimeObservation Expected");

		// change the ObsExpr with linked one
		newDurationExpression.getObsExpr().clear();
		newDurationExpression.getObsExpr().add(linkedLeft);
		newDurationExpression.getObsExpr().add(linkedRight);

		return newDurationExpression;
	}

	/**
	 * 
	 * @param jitter
	 * @param strategy
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	private ValueSpecification linkJitterExpression(JitterExpression jitter,
			IDecideStrategy strategy) throws TypeOrLinkException {
		// ( jitter( <instant-obs-expr> ) ) | ( jitter( <instant-obsexpr>
		// , <instant-obs-expr> ) )

		JitterExpression newJitter = VSLFactory.eINSTANCE
				.createJitterExpression();

		EList obsExpr = jitter.getObsExpr();
		// jitter can have one or two obs expr but they have to be instant.
		ObsCallExpression left = (ObsCallExpression) obsExpr.get(0);
		ObsCallExpression linkedLeft = linkObsCallExpression(left,
				UMLPackage.TIME_OBSERVATION, strategy);
		newJitter.getObsExpr().add(linkedLeft);

		if (obsExpr.size() > 1) {
			ObsCallExpression right = (ObsCallExpression) obsExpr.get(1);
			ObsCallExpression linkedRight = linkObsCallExpression(right,
					UMLPackage.TIME_OBSERVATION, strategy);
			newJitter.getObsExpr().add(linkedRight);
		}

		return newJitter;
	}

	public LiteralSpecification copyOf(LiteralSpecification ls) {

		if (ls instanceof LiteralDefault) {
			LiteralDefault newLs = VSLFactory.eINSTANCE.createLiteralDefault();
			return newLs;
		} else if (ls instanceof LiteralNull) {
			LiteralNull newLs = UMLFactory.eINSTANCE.createLiteralNull();
			return newLs;
		} else if (ls instanceof LiteralInteger) {
			LiteralInteger li = (LiteralInteger) ls;
			LiteralInteger newLi = UMLFactory.eINSTANCE.createLiteralInteger();
			newLi.setValue(li.getValue());
			return newLi;
		} else if (ls instanceof LiteralUnlimitedNatural) {
			LiteralUnlimitedNatural lun = (LiteralUnlimitedNatural) ls;
			LiteralUnlimitedNatural newLun = UMLFactory.eINSTANCE
					.createLiteralUnlimitedNatural();
			newLun.setValue(lun.getValue());
			return newLun;
		} else if (ls instanceof LiteralBoolean) {
			LiteralBoolean lb = (LiteralBoolean) ls;
			LiteralBoolean newLb = UMLFactory.eINSTANCE.createLiteralBoolean();
			newLb.setValue(lb.booleanValue());
			return newLb;
		} else if (ls instanceof LiteralDateTime) {
			LiteralDateTime ldt = (LiteralDateTime) ls;
			LiteralDateTime newLdt = VSLFactory.eINSTANCE
					.createLiteralDateTime();
			newLdt.setValue(ldt.getValue());
			return newLdt;
		} else if (ls instanceof LiteralReal) {
			LiteralReal lr = (LiteralReal) ls;
			LiteralReal newLr = VSLFactory.eINSTANCE.createLiteralReal();
			newLr.setValue(lr.getValue());
			return newLr;
		} else if (ls instanceof LiteralString) {
			LiteralString lstr = (LiteralString) ls;
			LiteralString newLstr = UMLFactory.eINSTANCE.createLiteralString();
			newLstr.setValue(lstr.getValue());
			return newLstr;
		}

		return null;
	}
}
