/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: TupleLinker.java,v 1.6 2007/10/15 09:30:05 fnizou Exp $
 */

package com.cea.nfp.parsers.modelgenerator;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.LiteralSpecification;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.StringExpression;
import org.eclipse.uml2.uml.ValueSpecification;

import VSL.TupleItemValue;
import VSL.TupleSpecification;
import VSL.VSLFactory;
import VSL.Variable;
import VSL.VariableCallExpression;

import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst;
import com.cea.nfp.parsers.texteditor.vsldatatypes.VSLComplexTypeUtil;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.BasicNfpType;

/**
 * Does the same thing as Link but for Tuple only. It is not intended to be used
 * elsewhere than Link.
 * 
 * @author T0081227 Francois NIZOU - 27 juil. 07
 * 
 */

public class TupleLinker {

	private IModelFacade facade;

	private Linker linker;

	/**
	 * 
	 * @param facade
	 * @param linker
	 */
	public TupleLinker(IModelFacade facade, Linker linker) {
		this.facade = facade;
		this.linker = linker;
	}

	/**
	 * 
	 * @param ts
	 * @param expected
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	public TupleSpecification linkTupleSpecification(TupleSpecification ts,
			DataType expected, IDecideStrategy strategy)
			throws TypeOrLinkException {

		if (isCompactForm(ts))
			return typeCompactTuple(ts, expected, strategy);
		if (expected != null)
			return typeForcedTuple(ts, expected, strategy);
		else
			return typeUnknownTuple(ts, strategy);
	}

	private TupleSpecification typeCompactTuple(TupleSpecification ts,
			DataType expected, IDecideStrategy strategy) throws TypeOrLinkException {
		EList tupleItems = ts.getTupleItem();
		if (tupleItems.size() != 2)
			throw new TupleException("cannot resolve tuple type");
		else {
			TupleItemValue unit = (TupleItemValue) tupleItems.get(1);
			unit.setTupleItemName(BasicNfpType.TUPLE_UNIT_FIELD);
			TupleItemValue valOrExp = (TupleItemValue) tupleItems.get(0);
			if (valOrExp.getItemValue() instanceof LiteralSpecification)
				valOrExp.setTupleItemName(BasicNfpType.TUPLE_VALUE_FIELD);
			else
				valOrExp.setTupleItemName(BasicNfpType.TUPLE_EXPR_FIELD);
			
			return linkTupleSpecification(ts, expected, strategy);
		}
	}

	/**
	 * This method is used when the expected datatype is unknown. In that case,
	 * we use a few hint (as unit, val, expr) to find out wich tuple the
	 * tupleSpecification does refers.
	 * 
	 * @param ts
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	protected TupleSpecification typeUnknownTuple(TupleSpecification ts,
			IDecideStrategy strategy) throws TypeOrLinkException {
		Hashtable<String, ValueSpecification> tupleItems = new Hashtable<String, ValueSpecification>();
		TupleSpecification result = null;
		for (Object o : ts.getTupleItem()) {
			TupleItemValue tiv = (TupleItemValue) o;
			tupleItems.put(tiv.getTupleItemName(), tiv.getItemValue());
		}

		// 1) find out if one of them is unit
		if (tupleItems.containsKey(BasicNfpType.TUPLE_UNIT_FIELD)) {
			ValueSpecification unitValue = tupleItems
					.get(BasicNfpType.TUPLE_UNIT_FIELD);
			Enumeration unit = resolveUnit(unitValue, strategy);
			if (unit != null) {
				ArrayList<DataType> tuplesType = resolveTupleFromUnit(unit);
				for (DataType type : tuplesType) {
					try {
						return typeForcedTuple(ts, type, strategy);
					} catch (Exception e) {

					}
				}

			}

		}

		if (tupleItems.containsKey(BasicNfpType.TUPLE_VALUE_FIELD)) {
			// find out if val is specified
			ValueSpecification linkedVSL = linker.link(tupleItems
					.get(BasicNfpType.TUPLE_VALUE_FIELD), null, strategy);
			DataType valueType = Typer.type(linkedVSL, facade).get(0);
			DataType type = resolveTupleFromValueType(valueType);
			return typeForcedTuple(ts, type, strategy);
		}

		if (tupleItems.containsKey(BasicNfpType.TUPLE_EXPR_FIELD)) {
			ValueSpecification linkedVSL = linker.link(tupleItems
					.get(BasicNfpType.TUPLE_EXPR_FIELD), null, strategy);
			DataType valueType = Typer.type(linkedVSL, facade).get(0);
			DataType type = resolveTupleFromValueType(valueType);
			return typeForcedTuple(ts, type, strategy);

		}

		result = resolveTupleGeneral(ts, strategy);

		return result;
	}

	/**
	 * This method is used when we know the Datatype. We then juste Type check
	 * it and link tupleItemValue.
	 * 
	 * @param ts
	 * @param expected
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	protected TupleSpecification typeForcedTuple(TupleSpecification ts,
			DataType expected, IDecideStrategy strategy)
			throws TypeOrLinkException {
		TupleSpecification result = VSLFactory.eINSTANCE
				.createTupleSpecification();

		String compositeType = VSLComplexTypeUtil.getCompositeType(expected);
		if (compositeType == null
				|| (!compositeType.equals(VSL.STEREOTYPE_TUPLE_TYPE))) {
			throw new TupleException(expected.getName() + " is not a Tuple");
		}
		Hashtable<String, Property> tupleItemDescr = new Hashtable<String, Property>();
		for (Object o : expected.getAllAttributes()) {
			Property prop = (Property) o;
			tupleItemDescr.put(prop.getName(), prop);
		}

		EList tupleItems = ts.getTupleItem();

		for (Object object : tupleItems) {
			TupleItemValue tupleItem = (TupleItemValue) object;
			String name = tupleItem.getTupleItemName();

			if (!tupleItemDescr.containsKey(name))
				throw new TupleException("the tuple " + expected.getName()
						+ " does not contain any tuple item called " + name);

			Property property = tupleItemDescr.get(name);
			DataType tupleItemDataType = facade.typeof(property);
			ValueSpecification linkedTupleItemValue = internalLink(tupleItem
					.getItemValue(), tupleItemDataType, tupleItems, strategy);

			if (!Typer
					.isTypeOf(linkedTupleItemValue, tupleItemDataType, facade))
				throw new TupleException("found type "
						+ Typer.type(linkedTupleItemValue, facade)
						+ " for tuple item " + name + " ("
						+ tupleItemDataType.getName() + " excepted)");

			TupleItemValue linkedTupleItem = createTupleItem(
					linkedTupleItemValue, property);

			result.getTupleItem().add(linkedTupleItem);
			tupleItemDescr.remove(property.getName());
		}

		// set default for the remaining property
		for (Property property : tupleItemDescr.values()) {
			TupleItemValue linkedTupleItem = createTupleItem(
					VSLFactory.eINSTANCE.createLiteralDefault(), property);
			result.getTupleItem().add(linkedTupleItem);
		}
		result.setType(expected);
		return result;
	}

	/**
	 * 
	 * @param itemValue
	 * @param tupleItemDataType
	 * @param tupleItems
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	private ValueSpecification internalLink(ValueSpecification itemValue,
			DataType tupleItemDataType, EList tupleItems,
			IDecideStrategy strategy) throws TypeOrLinkException {

		if (itemValue instanceof StringExpression) {
			StringExpression se = (StringExpression) itemValue;
			for (Object o : tupleItems) {
				TupleItemValue tupleItem = (TupleItemValue) o;
				ValueSpecification item = tupleItem.getItemValue();
				if (item instanceof Variable) {
					Variable var = (Variable) item;
					if (se.getName().equals(var.getName())) {
						Variable linkedvar = (Variable) linker.linkVariable(
								var, null);
						if (tupleItemDataType.getName().equals(
								Typer.type(linkedvar, facade).get(0).getName()))
							;
						VariableCallExpression callExpr = VSLFactory.eINSTANCE
								.createVariableCallExpression();
						callExpr.setDefiningVariable(linkedvar);
						callExpr.setType(tupleItemDataType);
						callExpr.setVariable(var.getName());
						return callExpr;
					}
				}
			}
		}

		return linker.link(itemValue, tupleItemDataType, strategy);

	}

	/**
	 * 
	 * @param value
	 * @param propery
	 * @return
	 */
	private TupleItemValue createTupleItem(ValueSpecification value,
			Property propery) {

		TupleItemValue tupleItemValue = VSLFactory.eINSTANCE
				.createTupleItemValue();
		tupleItemValue.setItemValue(value);
		tupleItemValue.setTupleAttribute(propery);
		tupleItemValue.setTupleItemName(propery.getName());
		return tupleItemValue;
	}

	private Enumeration resolveUnit(ValueSpecification unitValue,
			IDecideStrategy strategy) throws TypeOrLinkException {

		if (unitValue instanceof StringExpression) {
			ArrayList<EnumerationLiteral> enums = facade.getEnumerationsByName(
					unitValue.getName(), false);
			for (EnumerationLiteral literal : enums) {
				EList appliedStereotypes = literal.getAppliedStereotypes();
				for (Object object : appliedStereotypes) {
					Stereotype stereotype = (Stereotype) object;
					if (stereotype.getName().equals(MarteCst.STEREOTYPE_UNIT))
						return literal.getEnumeration();
				}
			}
		} else {
			ArrayList<DataType> types = Typer.type(linker.link(unitValue, null,
					strategy), facade);

			for (DataType type : types) {
				if (type instanceof Enumeration) {
					return (Enumeration) type;
				}
			}

		}

		return null;
	}

	/**
	 * Method attempting to resolv a tupleType when expected datatype is unknown
	 * and nor unit, value or unit attribute hve been found.
	 * 
	 * @param ts
	 * @param strategy
	 * @return
	 * @throws TypeOrLinkException
	 */
	private TupleSpecification resolveTupleGeneral(TupleSpecification ts,
			IDecideStrategy strategy) throws TypeOrLinkException {
		ArrayList<DataType> dataTypes = facade.getDataTypes();
		Hashtable<DataType, TupleSpecification> possibilities = new Hashtable<DataType, TupleSpecification>();
		for (DataType type : dataTypes) {
			String compositeType = VSLComplexTypeUtil.getCompositeType(type);
			if (compositeType != null
					&& compositeType.equals(VSL.STEREOTYPE_TUPLE_TYPE)) {
				try {

					TupleSpecification linked = typeForcedTuple(ts, type,
							strategy);
					possibilities.put(type, linked);
				} catch (TypeOrLinkException e) {
				}
			}
		}

		TupleSpecification result = null;
		if (possibilities.size() == 0) {
			throw new TupleException("Cannot resolve tuple");
		} else {
			// we search the most fitting type
			int min = Integer.MAX_VALUE;
			java.util.Enumeration<DataType> keys = possibilities.keys();
			while (keys.hasMoreElements()) {
				DataType key = (DataType) keys.nextElement();
				if (key.getAllAttributes().size() < min) {
					min = key.getAllAttributes().size();
					result = possibilities.get(key);
				}
			}
		}
		return result;
	}

	/**
	 * Find from the model every Datatype which has a property value of type
	 * valueType.
	 * 
	 * @param valueType
	 * @return
	 */
	private DataType resolveTupleFromValueType(DataType valueType) {
		ArrayList<DataType> dataTypes = facade.getDataTypes();
		ArrayList<DataType> possibleDataTypes = new ArrayList<DataType>();
		for (DataType type : dataTypes) {
			String compositeType = VSLComplexTypeUtil.getCompositeType(type);
			if (compositeType != null
					&& compositeType.equals(VSL.STEREOTYPE_TUPLE_TYPE)) {
				// this is a tuple
				EList allAttributes = type.getAllAttributes();
				for (Object object : allAttributes) {
					Property property = (Property) object;
					if (property.getName().equals(
							BasicNfpType.TUPLE_VALUE_FIELD)
							&& facade.typeof(property).getName().equals(
									valueType.getName()))
						possibleDataTypes.add(type);
				}
			}
		}

		if (possibleDataTypes.size() == 0)
			return null;

		DataType result = possibleDataTypes.get(0);

		// return the first datatype with no unit prop or the first one.
		for (DataType type : possibleDataTypes) {
			EList allAttributes = type.getAllAttributes();
			boolean hasUnit = false;
			for (Object object : allAttributes) {
				if (((Property) object).getName().equals(
						BasicNfpType.TUPLE_UNIT_FIELD))
					hasUnit = true;
			}
			if (!hasUnit) {
				result = type;
				break;
			}
		}

		return result;
	}

	/**
	 * Find out in the model the list of Datatype, stereotyped TupleType which
	 * has an unit attribute unit.
	 * 
	 * @param unit
	 * @return
	 */
	private ArrayList<DataType> resolveTupleFromUnit(Enumeration unit) {
		ArrayList<DataType> dataTypes = facade.getDataTypes();
		ArrayList<DataType> result = new ArrayList<DataType>();
		for (DataType type : dataTypes) {
			String compositeType = VSLComplexTypeUtil.getCompositeType(type);
			if (compositeType != null
					&& compositeType.equals(VSL.STEREOTYPE_TUPLE_TYPE)) {
				// this is a tuple
				EList allAttributes = type.getAllAttributes();
				for (Object object : allAttributes) {
					Property property = (Property) object;
					DataType typeofProp = facade.typeof(property);
					if (typeofProp == null)
						continue;

					if (property.getName()
							.equals(BasicNfpType.TUPLE_UNIT_FIELD)
							&& typeofProp.getName().equals(unit.getName()))
						result.add(type);
				}
			}
		}
		return result;
	}

	/**
	 * 
	 * @param ts
	 * @return true if ts is in compact form, e.g (10, s)
	 */
	protected boolean isCompactForm(TupleSpecification ts) {
		EList tupleItems = ts.getTupleItem();
		for (Object object : tupleItems) {
			TupleItemValue item = (TupleItemValue) object;
			String tupleItemName = item.getTupleItemName();
			if (tupleItemName == null || tupleItemName.trim().length() == 0)
				return true;
		}
		return false;
	}
}
