/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: Typer.java,v 1.5 2007/10/15 09:30:05 fnizou Exp $
 */

package com.cea.nfp.parsers.modelgenerator;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.LiteralBoolean;
import org.eclipse.uml2.uml.LiteralInteger;
import org.eclipse.uml2.uml.LiteralNull;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.LiteralUnlimitedNatural;
import org.eclipse.uml2.uml.Observation;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.TimeObservation;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;

import VSL.ChoiceSpecification;
import VSL.CollectionSpecification;
import VSL.ConditionalExpression;
import VSL.DurationExpression;
import VSL.EnumerationSpecification;
import VSL.InstantExpression;
import VSL.IntervalSpecification;
import VSL.LiteralDateTime;
import VSL.LiteralDefault;
import VSL.LiteralReal;
import VSL.ObsCallExpression;
import VSL.OperationCallExpression;
import VSL.PropertyCallExpression;
import VSL.TupleItemValue;
import VSL.TupleSpecification;
import VSL.Variable;
import VSL.VariableCallExpression;

import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst;
import com.cea.nfp.parsers.texteditor.vsldatatypes.VSLComplexTypeUtil;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.BasicNfpType;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.PrimitivesTypes;

/**
 * This class is used to type any VSL Expression.
 * 
 * @author T0081227 Francois NIZOU - 7 aot 07
 * @author Sbastien Demathieu 05/11/07 (Fixed NullPointerException error)
 * 
 */
public class Typer {

	private static final String Type_not_found_error_msg = "could not find reference to ";

	/**
	 * Return the list of possible datatype for the VSL vsl. The list might
	 * contains more than one element if the vsl element is ambigous. Note: the
	 * vsl must have been Linked first.
	 * {@link Linker#link(ValueSpecification, DataType)}
	 * 
	 * @param vsl
	 * @param facade
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	public static ArrayList<DataType> type(ValueSpecification vsl,
			IModelFacade facade) throws TypeOrLinkException {
		ArrayList<DataType> result = new ArrayList<DataType>();

		// SD Fix (05/11/07): previous code did not take into account that the
		// ValueSpecification can be null if a complex expression is being
		// edited.
		if ((vsl != null) && (vsl.getType() != null)) {
			result.add((DataType) vsl.getType());
			return result;
		}

		ArrayList<DataType> datatypes = new ArrayList<DataType>();

		if (vsl instanceof LiteralBoolean) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.BOOLEAN);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.BOOLEAN);
			result.add(datatypes.get(0));
		} else if (vsl instanceof LiteralInteger) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.INTEGER);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.INTEGER);
			result.add(datatypes.get(0));
		} else if (vsl instanceof LiteralDateTime) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.DATETIME);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.DATETIME);
			result.add(datatypes.get(0));
		} else if (vsl instanceof LiteralReal) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.REAL);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.REAL);
			result.add(datatypes.get(0));
		} else if (vsl instanceof LiteralNull)
			return null;
		else if (vsl instanceof LiteralString) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.STRING);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.STRING);
			result.add(datatypes.get(0));
		} else if (vsl instanceof LiteralUnlimitedNatural) {
			datatypes = facade
					.getDataTypesByName(PrimitivesTypes.UNLIMITEDNATURAL);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.UNLIMITEDNATURAL);
			result.add(datatypes.get(0));
		} else if (vsl instanceof Variable)
			result.add((DataType) ((Variable) vsl).getType());
		else if (vsl instanceof VariableCallExpression)
			result.add((DataType) ((VariableCallExpression) vsl)
					.getDefiningVariable().getType());
		else if (vsl instanceof PropertyCallExpression) {
			Property prop = ((PropertyCallExpression) vsl)
					.getDefiningProperty();
			DataType datatype = facade.typeof(prop);
			if (datatype != null)
				result.add(datatype);
			else
				throw new TypeOrLinkException("the property '" + prop.getName()
						+ "' is not typed by a DataType");
		} else if (vsl instanceof ConditionalExpression) {
			datatypes = facade.getDataTypesByName(PrimitivesTypes.BOOLEAN);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ PrimitivesTypes.BOOLEAN);
			result.add(datatypes.get(0));
		} else if (vsl instanceof OperationCallExpression) {
			Operation operation = ((OperationCallExpression) vsl)
					.getDefiningOperation();
			Type type = facade.typeof(operation.getReturnResult());
			if (type instanceof DataType)
				result.add((DataType) type);
			else
				throw new BadTypeException(type);
		} else if (vsl instanceof TupleSpecification) {
			return typeTuple((TupleSpecification) vsl, facade);
		} else if (vsl instanceof IntervalSpecification) {
			return null;
		} else if (vsl instanceof ChoiceSpecification) {
			result.add(typeChoice((ChoiceSpecification) vsl, facade));
		} else if (vsl instanceof EnumerationSpecification) {
			EnumerationSpecification es = (EnumerationSpecification) vsl;
			result.add(es.getEnumLiteral().getEnumeration());
		} else if (vsl instanceof DurationExpression) {
			datatypes = facade.getDataTypesByName(BasicNfpType.NFP_DURATION);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ BasicNfpType.NFP_DURATION);
			DataType nfpDuration = datatypes.get(0);
			result.add(nfpDuration);
		} else if (vsl instanceof InstantExpression) {
			datatypes = facade.getDataTypesByName(BasicNfpType.NFP_DATE_TIME);
			if (datatypes.size() == 0)
				throw new TypeOrLinkException(Type_not_found_error_msg
						+ BasicNfpType.NFP_DATE_TIME);
			DataType nfpDuration = datatypes.get(0);
			result.add(nfpDuration);
		} else if (vsl instanceof ObsCallExpression) {
			Observation innerObs = ((ObsCallExpression) vsl).getObservation();
			if (innerObs instanceof TimeObservation) {
				datatypes = facade
						.getDataTypesByName(BasicNfpType.NFP_DATE_TIME);
				if (datatypes.size() == 0)
					throw new TypeOrLinkException(Type_not_found_error_msg
							+ BasicNfpType.NFP_DATE_TIME);
				DataType nfpDuration = datatypes.get(0);
				result.add(nfpDuration);
			} else {
				datatypes = facade
						.getDataTypesByName(BasicNfpType.NFP_DURATION);
				if (datatypes.size() == 0)
					throw new TypeOrLinkException(Type_not_found_error_msg
							+ BasicNfpType.NFP_DURATION);
				DataType nfpDuration = datatypes.get(0);
				result.add(nfpDuration);
			}
		}

		return result;
	}

	/**
	 * 
	 * @param specification
	 * @param facade
	 * @return
	 * @throws BadTypeException
	 * @throws UnresolvedNameException
	 */
	private static DataType typeChoice(ChoiceSpecification choice,
			IModelFacade facade) throws TypeOrLinkException {
		String chosenAlternative = choice.getChosenAlternative();
		ValueSpecification value = choice.getValue();
		ArrayList<DataType> valueTypes = type(value, facade);
		if (valueTypes == null || valueTypes.size() == 0)
			return null;

		for (DataType valueType : valueTypes) {

			DataType choiceType = facade
					.getChoice(chosenAlternative, valueType);
			if (choiceType == null)
				continue;
			return choiceType;

		}
		throw new UnresolvedNameException(chosenAlternative);
	}

	/**
	 * 
	 * @param tuple
	 * @param facade
	 * @return
	 */
	protected static ArrayList<DataType> typeTuple(TupleSpecification tuple,
			IModelFacade facade) {
		ArrayList<String> tupleItemsName = new ArrayList<String>();
		EList tupleItems = tuple.getTupleItem();
		for (Object object : tupleItems) {
			String itemName = ((TupleItemValue) object).getTupleItemName();
			tupleItemsName.add(itemName);
		}
		ArrayList<DataType> mathingDatatype = facade
				.findTupleByItemNames(tupleItemsName);
		if (mathingDatatype.size() > 0)
			return mathingDatatype;
		else
			return null;
	}

	/**
	 * 
	 * @param linkedTupleItem
	 * @param tupleItemDataType
	 * @return
	 * @throws TypeOrLinkException
	 */
	public static boolean isTypeOf(ValueSpecification vsl, DataType datatype,
			IModelFacade facade) throws TypeOrLinkException {

		if (vsl instanceof LiteralDefault || vsl instanceof LiteralNull
				|| datatype.getName().equals(PrimitivesTypes.VSLEXPRESSION))
			return true;

		String complexTypeName = VSLComplexTypeUtil.getCompositeType(datatype);
		if (complexTypeName != null
				&& complexTypeName
						.equals(MarteCst.VSL.STEREOTYPE_COLLECTION_TYPE)) {
			return (vsl instanceof CollectionSpecification);
		} else if (complexTypeName != null
				&& complexTypeName
						.equals(MarteCst.VSL.STEREOTYPE_INTERVAL_TYPE)) {
			return (vsl instanceof IntervalSpecification);
		}

		else {

			DataType vslType = type(vsl, facade).get(0);
			return (vslType.getName().equals(datatype.getName()));
		}
	}

}
