/*******************************************************************************
 * Copyright (c) 2007 CEA List, THALES.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Eclipse Public License for more details.
 *
 * Contributors:
 *     CEA List - initial API and implementation
 *     THALES   - Modified the code to use IModelFacade for code access
 *     			- Modified parseAndModifyProperty so the editor accept expression
 *     			for primitive or complete type
 *      		- Modified code to correct or add behaviors
 *******************************************************************************/

package com.cea.nfp.parsers.modelgenerator;

/**
 * @author T0081227 Francois NIZOU
 */

import java.io.BufferedReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;

import antlr.NoViableAltException;
import antlr.RecognitionException;
import antlr.TokenStreamException;

import com.cea.nfp.parsers.antlr.VSLLexer;
import com.cea.nfp.parsers.antlr.VSLParser;
import com.cea.nfp.parsers.texteditor.vsldatatypes.IContext;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst;
import com.cea.nfp.parsers.texteditor.vsldatatypes.VSLComplexTypeUtil;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.PrimitivesTypes;

public class VSLGenerator {
	private Property property;

	private DataType subdatatype;

	private int context;

	private IModelFacade facade = null;

	private ValueSpecification vsl = null;

	/**
	 * Default Constructor
	 * 
	 * @param property
	 *            the property to modify
	 */

	// Modified by THALES,
	// Francois 31 2007: this class does not directly refer to the opaque
	// expression, nor the datatype.
	public VSLGenerator(IModelFacade facade) {
		this.facade = facade;
	}

	/**
	 * 
	 * @param text
	 * @param datatype
	 * @return
	 */
	// Added by Thales
	// Francois 20 juillet, this method is more general than
	// parseAndvalidatePorperty,
	public String parseAndValidateTypedExpression(String text, DataType datatype) {

		String message = null;
		VSLLexer lexer = new VSLLexer(
				new BufferedReader(new StringReader(text)));
		VSLParser parser = new VSLParser(lexer);
		parser.setValidation(true);
		try {
			ValueSpecification parsedVsl = parser.valueSpecification();

			// If the parser has not read all, the text in editor is incorrect.
			String hasReadAll = parser.notParsedEndMessage();
			if (hasReadAll != null) {
				message = "unexpected characters: " + hasReadAll;
			} else {
				Linker linker = new Linker(facade, new BasicDecideStrategy());
				this.vsl = linker.link(parsedVsl, datatype);
			}
			this.context = parser.getContext();
		} catch (NoViableAltException e) {
			this.context = parser.getContext();
			if (e.token.getText() == null)
				message = "unexpected end of text at char " + e.column;
			else
				message = e.getLocalizedMessage();

		} catch (RecognitionException e) {
			this.context = parser.getContext();
			message = e.getLocalizedMessage();

		} catch (TokenStreamException e) {
			this.context = parser.getContext();
			message = e.getLocalizedMessage();
		} catch (UnresolvedNameException e) {
			this.context = parser.getContext();
			message = e.getMessage() + " cannot be resolved";
		} catch (BadTypeException e) {
			this.context = parser.getContext();
			message = e.getMessage();
		} catch (TypeOrLinkException e) {
			this.context = parser.getContext();
			message = e.getMessage();
		}

		return message;

	}

	/**
	 * Parse the label of the property and modify property attributes.
	 * 
	 * @param label
	 *            the label that defines the property
	 * @throws RecognitionException
	 * @throws TokenStreamException
	 */
	// unused
	public void parseAndModifyProperty(String label)
			throws RecognitionException, TokenStreamException {
		VSLLexer lexer = new VSLLexer(new StringReader(label));
		VSLParser parser = new VSLParser(lexer);
		parser.valueSpecification();
	}

	/**
	 * Parse the label of the property and validate it.
	 * 
	 * @param label
	 *            the label that defines the property
	 * @return null if label is valid, else return the message that describes
	 *         the error.
	 */

	// Modified by THALES
	// Francois, july 31 2007:
	// - parser now handler white space as separator, therefor, no need
	// to trim the text anymore.
	// - Behavior of this method changed, many handlers added
	public String parseAndValidateProperty(String text, DataType datatype) {
		String message = null;

		// to validate the message : parse it. If no errors, it is ok. If
		// exceptions : not ok. Returns the exception message
		// VSLLexer lexer = new VSLLexer(new BufferedReader(new

		VSLLexer lexer = new VSLLexer(
				new BufferedReader(new StringReader(text)));

		VSLParser parser = new VSLParser(lexer);
		parser.setValidation(true);

		ValueSpecification parserVSL = null;

		try {

			if (datatype.getName().equals(PrimitivesTypes.BOOLEAN)) {
				parserVSL = parser.booleanLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.DATETIME)) {
				parserVSL = parser.datetimeLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.NFPEXPRESSION)) {
				parserVSL = parser.valueSpecification();
			}
			if (datatype.getName().equals(PrimitivesTypes.REAL)) {
				parserVSL = parser.realLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.STRING)) {
				parserVSL = parser.stringLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.INTEGER)) {
				parserVSL = parser.integerLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.UNLIMITEDNATURAL)) {
				parserVSL = parser.unlimitedLiteral();
			}
			if (datatype.getName().equals(PrimitivesTypes.VSLEXPRESSION)) {
				parserVSL = parser.valueSpecification();
			}
			if (datatype instanceof Enumeration) {
				parserVSL = parser.enumerationSpecification();
			}

			// vsl types
			String compositeTypeName = VSLComplexTypeUtil
					.getCompositeType(datatype);
			if (compositeTypeName != null) {
				if (compositeTypeName.equals(VSL.STEREOTYPE_CHOICE_TYPE)) {
					parserVSL = parser.choice();
				} else if (compositeTypeName
						.equals(VSL.STEREOTYPE_COLLECTION_TYPE)) {
					parserVSL = parser.collection();
				} else if (compositeTypeName.equals(VSL.STEREOTYPE_TUPLE_TYPE)) {
					parserVSL = parser.tuple();
				} else if (compositeTypeName
						.equals(VSL.STEREOTYPE_INTERVAL_TYPE)) {
					parserVSL = parser.interval();
				}
			}

			this.context = parser.getContext();

		} catch (Exception e) {
			this.context = parser.getContext();
			message = e.getLocalizedMessage();

			if (text != null && text.trim().length() > 0)
				message = this.parseAndValidateTypedExpression(text, datatype);

		}

		try {
			// Francois july 30
			// we check if antlr has read all. If not we force a complete parse.
			if (parser.notParsedEndMessage() != null) {
				message = this.parseAndValidateTypedExpression(text, datatype);
				parserVSL = null;
			} else {
				Linker linker = new Linker(facade, new BasicDecideStrategy());
				this.vsl = linker.link(parserVSL, datatype);
			}
		} catch (TokenStreamException e) {
		} catch (TypeOrLinkException e) {
			message = e.getMessage();
		}

		return message;
	}

	/**
	 * 
	 * @param text
	 * @param datatype
	 * @param offset
	 * @return
	 */
	public String isCollectionValid(String text, DataType datatype, int offset) {
		String message = "";
		String tmp = "";

		message = parseAndValidateProperty(text, datatype);

		Stereotype stereotype = null;

		EList appliedStereotypes = datatype.getAppliedStereotypes();
		for (Object object : appliedStereotypes) {
			Stereotype s = (Stereotype) object;
			if (s.getName().equals("CollectionType"))
				;
			stereotype = s;
		}

		if (stereotype == null) {
			message = "the expected datatype is not a collection";
		}

		Property collectionAttrib = null;

		if (stereotype != null)
			collectionAttrib = (Property) datatype.getValue(stereotype,
					VSL.VSL_COLLECTION_ATTRIB);

		if (collectionAttrib != null) {
			subdatatype = (DataType) collectionAttrib.getType();
		} else {
			System.err
					.println("collectionAttrib property should be defined for "
							+ datatype.getName() + " DataType!");
		}

		if (offset > 0 && offset < text.length()) {

			tmp = parseAndValidateProperty("", subdatatype);

			if (tmp != null) {
				message = message + tmp + "; ";
			}
		}

		return message;
	}

	/**
	 * 
	 * @param text
	 * @param datatype
	 * @param offset
	 * @return
	 */

	// Modified by THALES
	// francois: modified the dataType check. The spec specify an intervalType
	// has an unique property called IntervalAttrib. Only IntervalSpecification
	// has a min & max.
	public String isIntervalValid(String text, DataType datatype, int offset) {
		String message = "";
		String tmp = "";

		message = parseAndValidateProperty(text, datatype);

		Stereotype stereotype = null;
		EList appliedStereotypes = datatype.getAppliedStereotypes();
		for (Object object : appliedStereotypes) {
			Stereotype s = (Stereotype) object;
			if (s.getName().equals("IntervalType"))
				;
			stereotype = s;
		}

		if (stereotype == null) {
			message = "the expected datatype is not an Interval";
		}

		Property intervalAttribute = (Property) datatype.getValue(stereotype,
				VSL.VSL_INTERVAL_ATTRIB);
		subdatatype = facade.typeof(intervalAttribute);

		int dotPos = text.indexOf("..");

		if (offset > 0 && (offset <= dotPos || offset > dotPos + 1)) {

			tmp = parseAndValidateProperty("", subdatatype);

			if (tmp != null) {
				message = message + tmp + "; ";
			}
		}

		return message;
	}

	/**
	 * 
	 * @param text
	 * @param datatype
	 * @param offset
	 * @return
	 */
	// Modified By THALES
	// Francois july 10 2007:
	// Modified the code checking the type constraint of the choice value.
	// Completion works fine too.
	public String isChoiceValid(String text, DataType datatype, int offset) {
		String message = null;
		message = parseAndValidateProperty(text, datatype);

		if (text == null | text.length() == 0)
			return message;

		// if (message != null && message.length() > 0)
		// return message;

		int lParenPos = text.indexOf("(");
		int rParenPos = text.lastIndexOf(")");
		Type alternativeType = null;
		// Property selectedAlternative = null;

		if (text != "" && lParenPos >= 0) {
			String alternativeName = text.substring(0, lParenPos);

			Iterator<Property> iter = datatype.getAllAttributes().iterator();
			while (iter.hasNext()) {
				Property prop = iter.next();
				if (prop.getName().equals(alternativeName))
					alternativeType = prop.getType();
			}
		}

		// last update, 02/11/07
		// code was throwing unhandled exception when no perentheses were found
		if (lParenPos == -1) {
			message = "Incorrect choice, missing '('";
			return message;
		}

		if (rParenPos == -1) {
			message = "Incorrect choice, missing ')'";
			return message;
		}
		// end of update

		String choiceValue = text.substring(lParenPos + 1, rParenPos);

		String valueParseMessage = null;

		// Last update 02/11/07
		// unhandled exception was thrown when choice alternative was not
		// correct
		if (alternativeType == null) {
			message = "cannot not resolve choice";
			return message;
		}
		// end of update

		String compositeType = VSLComplexTypeUtil
				.getCompositeType((DataType) alternativeType);
		if (compositeType != null
				&& compositeType.equals(VSL.STEREOTYPE_TUPLE_TYPE)) {
			// TODO: offset is incorrect as we modified initial String.
			// TupleSPecifc parsing is therefore buggy.
			// Bug n 1063 (was: 1062, changed by S. Demathieu)
			
			valueParseMessage = isTupleValid("(" + choiceValue + ")",
					(DataType) alternativeType, offset);
		} else {
			valueParseMessage = parseAndValidateProperty(choiceValue,
					(DataType) alternativeType);
		}

		if (valueParseMessage != null)
			valueParseMessage = "the choice value is not correct, reason: "
					+ valueParseMessage + " (type expected is "
					+ alternativeType.getName() + ")";

		return valueParseMessage;
	}

	/**
	 * 
	 * @param text
	 * @param datatype
	 * @param offset
	 * @return
	 */
	// Modified by THALES,
	// Francois, july 30 2007. Changed this method so it won't force user to
	// enter
	// all the field. For instance, (value=1, unit=s) is now correct for a
	// NFP_Duration.
	public String isTupleValid(String text, DataType datatype, int offset) {
		String message = "";
		String[] textSplit;
		String tmp = "";
		int auxPosition = 0;
		String auxText[];

		// textSplit = text.split(",");
		textSplit = tupleSplit(text);
		if (textSplit.length > 0) {
			tmp = "";
			if (textSplit[0].length() > 0) {
				tmp = textSplit[0].substring(0, 1);
			}
			if (!tmp.equals("(")) {
				message = "expected '('; ";
			} else {

				textSplit[0] = textSplit[0].substring(1, textSplit[0].length());
				// Francois, july 10 2007,
				// removed the -1 as it was incorrect in some case.
			}
			tmp = "";
			if (textSplit[0].length() > 0) {
				if (textSplit[textSplit.length - 1].length() > 0)
					tmp = textSplit[textSplit.length - 1].substring(
							textSplit[textSplit.length - 1].length() - 1,
							textSplit[textSplit.length - 1].length());
				else
					tmp = "";
			}
			if (!tmp.equals(")")) {
				message = message + "expected ')'; ";
			} else {
				textSplit[textSplit.length - 1] = textSplit[textSplit.length - 1]
						.substring(0,
								textSplit[textSplit.length - 1].length() - 1);
			}

			Hashtable<String, Property> properties = new Hashtable<String, Property>();
			for (int i = 0; i < datatype.getAllAttributes().size(); i++) {
				property = (Property) datatype.getAllAttributes().get(i);
				properties.put(property.getName(), property);
			}
			for (int i = 0; i < textSplit.length; i++) {
				int indexOfEq = textSplit[i].indexOf("=");
				if (indexOfEq != -1) {
					String fieldName = textSplit[i].substring(0, indexOfEq)
							.trim();
					if (properties.containsKey(fieldName)) {

					} else {
						message = message + "the tuple " + datatype.getName()
								+ " does not contains field called "
								+ fieldName + ";";
					}
				} else {
					message = message + " found '" + textSplit[i]
							+ "' instead of tupleItem ;";
				}
			}

			if (message.equals("")) {
				if (offset > 0) { // called by content assist
					auxText = (text.substring(0, offset)).split(",");
					auxPosition = auxText.length - 1;
					String currentFieldStr = auxText[auxPosition].replaceAll(
							"[)]", "").replaceAll("[(]", "");
					int indexOfEq = currentFieldStr.indexOf("=");

					// Workaround, 02/11/07 to be solved.
					if (indexOfEq == -1) {
						message = "";
						return message;
					}
					// end workaround

					String fieldName = currentFieldStr.substring(0, indexOfEq)
							.trim();
					String fieldValue = currentFieldStr
							.substring(indexOfEq + 1).trim();

					property = properties.get(fieldName);
					subdatatype = (DataType) property.getType();
					tmp = parseAndValidateProperty(fieldValue, subdatatype);

					if (tmp != null) {
						message = message + tmp + "; ";
					}
				} else { // called by checker
					for (int i = 0; i < textSplit.length; i++) {

						int indexOfEq = textSplit[i].indexOf("=");
						String fieldName = textSplit[i].substring(0, indexOfEq)
								.trim();
						String fieldValue = textSplit[i].substring(
								indexOfEq + 1).trim();
						property = properties.get(fieldName);
						subdatatype = (DataType) property.getType();
						tmp = parseAndValidateProperty(fieldValue, subdatatype);

						if (tmp != null) {
							message = message + tmp + "; ";
						}
					}
				}

			} else {
				String fullParse = parseAndValidateProperty(text, datatype);

				message = (fullParse == null || fullParse.trim().length() == 0) ? ""
						: fullParse;
			}

		} else {
			String fullParse = parseAndValidateProperty(text, datatype);

			message = (fullParse == null || fullParse.trim().length() == 0) ? ""
					: fullParse;
		}
		if (message.equals("")) {
			return null;
		} else {
			// francois, july 30 2007: allow completion on tuple item name.
			// If we re in completion mode and the cursor is somewhere where
			// user might want to
			// add a new name=value, we force the context in tupleItem.
			if (offset > 0) {
				int indexOfLeftParen = text.indexOf("(");
				int indexOfComa = text.lastIndexOf(",", offset);
				int indexOfNextComa = text.indexOf(",", offset);
				int indexOfRightParen = text.indexOf(")");
				if (indexOfLeftParen != -1
						&& text.substring(indexOfLeftParen + 1, offset).trim()
								.length() == 0) {
					this.context = IContext.TUPLEITEM;
				} else if (indexOfComa != -1
						&& indexOfNextComa != -1
						&& text.substring(indexOfComa, indexOfNextComa - 1)
								.trim().length() == 0) {
					this.context = IContext.TUPLEITEM;
				} else if (indexOfComa != -1
						&& indexOfRightParen != -1
						&& text.substring(indexOfComa, indexOfRightParen - 1)
								.trim().length() == 0) {
					this.context = IContext.TUPLEITEM;
				}
			}
			return message;
		}
	}

	private String[] tupleSplit(String tuple) {
		ArrayList<String> results = new ArrayList<String>();
		String tmp = "";
		int count = 0;
		if (tuple.length() > 0) {
			if (tuple.charAt(0) == '(') {
				tmp = "(";
				char[] chars = tuple.toCharArray();
				for (int i = 1; i < chars.length; i++) {
					if (chars[i] == ',' && count == 0) {
						results.add(tmp);
						tmp = "";
					} else
						tmp += chars[i];
					if (chars[i] == '(')
						count++;
					if (chars[i] == ')')
						count--;
				}
				if (tmp != null)
					results.add(tmp);
			}
		} else {
			results.add("");
		}

		return results.toArray(new String[0]);

	}

	public int getContext() {
		return this.context;
	}

	public DataType getSubDataType() {
		return subdatatype;
	}

	/**
	 * 
	 * @return the Correct last VSL parsed.
	 */
	public ValueSpecification getVsl() {
		return vsl;
	}

}