/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: UMLModelFacade.java,v 1.8 2007/10/15 09:30:05 fnizou Exp $
 */
package com.cea.nfp.parsers.uml;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Observation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.TypedElement;
import org.eclipse.uml2.uml.resource.UMLResource;

import VSL.Variable;

import com.cea.nfp.parsers.modelgenerator.IModelFacade;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;

/**
 * The model facade for UML Eclipse.
 * 
 * @author T0081227 Francois NIZOU - 6 aot 07
 * 
 */
public class UMLModelFacade implements IModelFacade {

	protected static Hashtable<String, DataType> marteLibDatatype = null;

	protected DatatypeCollection datatypesCollection;

	protected ObservationCollection observationCollection;

	protected PropertyCollection propertyCollection;

	protected VariablesCollection variablesCollection;

	private Model model;

	private Element vslDefiningElement;

	public UMLModelFacade(Model model, Element vslDefiningElement) {
		this.model = model;
		this.vslDefiningElement = vslDefiningElement;
		ArrayList<Model> libraries = new ArrayList<Model>();
		ResourceSet resourceSet = model.eResource().getResourceSet();
		for (Object o : resourceSet.getResources()) {
			if (o instanceof UMLResource) {
				UMLResource res = (UMLResource) o;
				Object object = res.getContents().get(0);
				if (object instanceof Model) {
					Model m = (Model) object;
					if (m.isModelLibrary())
						libraries.add(m);
				}
			}
		}
		this.datatypesCollection = new DatatypeCollection(libraries, model);
		this.observationCollection = new ObservationCollection(model);
		this.propertyCollection = new PropertyCollection(model, this);
		this.variablesCollection = new VariablesCollection(model, this,
				vslDefiningElement);
	}

	public DataType typeof(TypedElement typedElement) {
		Type type = typedElement.getType();
		if (type instanceof DataType)
			return (DataType) type;
		else
			return null;
	}

	public ArrayList<DataType> getDataTypes() {
		return datatypesCollection.getAllDatatypes();
	}

	public ArrayList<DataType> getDataTypesByName(String name) {
		return datatypesCollection.getDatatypesByName(name);
	}

	public ArrayList<DataType> getDataTypesStartWith(String startwith) {
		return datatypesCollection.getDatatypesStartWith(startwith);
	}

	public ArrayList<Observation> getObservations() {
		return observationCollection.getObservations();
	}

	public ArrayList<Observation> getObservationsByName(String name) {
		return observationCollection.getObservationsByName(name);
	}

	public ArrayList<Observation> getObservationsStartWith(String prefix) {
		return observationCollection.getObservationsStartWith(prefix);
	}

	public ArrayList<Property> getProperties() {
		return propertyCollection.getProperties();
	}

	public ArrayList<Property> getpropertiesByName(String namespace, String name) {
		return propertyCollection.getpropertiesByName(namespace, name);
	}

	public ArrayList<Property> getPropertiesStartWith(String prefix) {
		return propertyCollection.getPropertiesStartWith(prefix);
	}

	/**
	 * 
	 * @return All the VSL variables of the Model.
	 */
	public ArrayList<Variable> getVariables() {
		return variablesCollection.getVariables();
	}

	/**
	 * 
	 * @param prefix
	 * @return all the VSL variables of the Model Starting with startwith
	 */
	public ArrayList<Variable> getVariablesStartWith(String prefix) {
		return variablesCollection.getVariablesStartWith(prefix);
	}

	/**
	 * 
	 * @param namespace
	 * @param name
	 * @return all Variables called namespace.name
	 */
	public ArrayList<Variable> getVariablesByName(String namespace, String name) {
		return variablesCollection.getVariablesByName(namespace, name);
	}

	public ArrayList<EnumerationLiteral> getEnumerationsByName(String name,
			boolean isStartWith) {
		ArrayList<EnumerationLiteral> result = new ArrayList<EnumerationLiteral>();

		ArrayList<DataType> allDatatypes = datatypesCollection
				.getAllDatatypes();
		for (DataType type : allDatatypes) {
			if (type instanceof Enumeration) {
				Enumeration enum_ = (Enumeration) type;
				EList ownedLiterals = enum_.getOwnedLiterals();
				for (Object object : ownedLiterals) {
					EnumerationLiteral enumerationLiteral = (EnumerationLiteral) object;
					if (isStartWith
							&& enumerationLiteral.getName().startsWith(name)) {
						result.add(enumerationLiteral);
						break;
					} else if (enumerationLiteral.getName().equals(name)) {
						result.add(enumerationLiteral);
						break;
					}
				}
			}
		}

		return result;
	}

	public String getCurrentContextName() {
		Element elt = vslDefiningElement;
		while (elt != null && !(elt instanceof NamedElement)) {
			elt = elt.getOwner();
		}

		if (elt == null)
			return "";

		NamedElement ne = (NamedElement) elt;
		NamedElement tmp = ne;
		String ns = "";
		while (!(tmp.getNamespace() instanceof Model)) {
			tmp = tmp.getNamespace();
			if (ns.length() == 0)
				ns += tmp.getName();
			else
				ns += tmp.getName() + "." + ns;
		}
		return ns;
	}

	public ArrayList<DataType> findTupleByItemNames(
			ArrayList<String> tupleItemNames) {
		return datatypesCollection.findTupleByItemNames(tupleItemNames);
	}

	public DataType getChoice(String chosenAlternative, DataType valueType) {
		return datatypesCollection.getChoice(chosenAlternative, valueType);
	}

	public String getQualifiedName(NamedElement element) {
		String name = element.getName();
		Element father = (Element) element.getOwner();
		while (father.getOwner() != null) {
			name = ((NamedElement) father).getName() + "." + name;
			father = father.getOwner();
		}
		return name;
	}

	public boolean isLocalProperty(Property property) {
		Element elt = vslDefiningElement;
		NamedElement ne = null;
		while (elt != null && !(elt instanceof NamedElement)) {
			elt = elt.getOwner();
		}

		if (elt == null)
			return false;
		else {
			ne = (NamedElement) elt;
			return (property.getNamespace().equals(ne.getNamespace()));
		}
	}

	/**
	 * 
	 * @param owner
	 * @return
	 */
	public String getContextFromElement(Element owner) {
		if (owner instanceof Model)
			return "";
		else if (isStereotypedBy(owner, VSL.STEREOTYPE_EXPRESSION_CONTEXT) != null) {
			return getQualifiedName((NamedElement) owner);
		} else {
			return getContextFromElement(owner.getOwner());
		}
	}

	protected Stereotype isStereotypedBy(Element elt, String stereotypeName) {
		EList appliedStereotypes = elt.getAppliedStereotypes();
		for (Object object : appliedStereotypes) {
			Stereotype s = (Stereotype) object;
			if (s.getName().equals(stereotypeName))
				return s;
		}
		return null;
	}

	public String getCurrentVariableContextName() {
		return getContextFromElement(vslDefiningElement);
	}

}
