/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 *
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>

 *
 * $Id: UMLModel.java,v 1.4 2007/10/15 09:30:45 fnizou Exp $
 */
package model;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.ValueSpecification;

import com.cea.nfp.parsers.modelgenerator.IModelFacade;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.PrimitivesTypes;
import com.cea.nfp.parsers.uml.UMLModelFacade;
import com.cea.nfp.popup.actions.EclipseUMLResourceFacade;
import com.cea.nfp.popup.actions.VSLLabelEditorDialog;

/**
 * 
 * @author T0081227 Francois NIZOU - july 6 2007
 * 
 */
public class UMLModel {

	private UMLModel() {

	}

	private static UMLModel self = null;

	public static UMLModel instance() {
		if (self == null)
			self = new UMLModel();
		return self;
	}

	/**
	 * Open an editor on a VSL expression
	 * 
	 * @param opaqueExpression
	 * @param datatype
	 */
	public void openEditor(OpaqueExpression opaqueExpression, DataType datatype) {
		Shell shell = new Shell();

		EclipseUMLResourceFacade vslfacade = new EclipseUMLResourceFacade(
				opaqueExpression, datatype);
		IModelFacade facade = new UMLModelFacade(opaqueExpression.getModel(),
				(Element) opaqueExpression.eContainer());
		VSLLabelEditorDialog pled = new VSLLabelEditorDialog(shell, vslfacade,
				facade);
		pled.open();
	}

	/**
	 * Ask the opened Editor to save current model
	 * 
	 */
	public void saveModel() {
		IWorkbenchWindow activeWorkbenchWindow = PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow();
		activeWorkbenchWindow.getActivePage().getActiveEditor().doSave(
				new NullProgressMonitor());
	}

	/**
	 * 
	 * @param constraint
	 */
	public void handleConstraint(Constraint constraint) {

		Shell shell = new Shell();

		// althought the common use case is for NFP_Constraint,
		// we allow user to open the editor on any constraint.

		// if (!isNFPConstraint(constraint)) {
		//
		// MessageDialog
		// .openError(shell, "VSL Editor Error",
		// "VSL Edition is only available for Constraint stereotyped
		// <<NfpConstraint>>");
		// return;
		// }

		ValueSpecification specification = constraint.getSpecification();
		OpaqueExpression expression = null;

		// get opaque expression or create it
		if (specification == null) {
			expression = UMLFactory.eINSTANCE.createOpaqueExpression();

			constraint.setSpecification(expression);
		} else {
			if (specification instanceof OpaqueExpression) {
				expression = (OpaqueExpression) specification;
			} else {
				// some specification already exist but is not opaqueExpression
				// we prompt user
				boolean openConfirm = MessageDialog
						.openConfirm(
								shell,
								"VSL Edition",
								"Another specification that is not VSL does exist for this constraint, do you want to replace it?");
				if (!openConfirm)
					return;

				// user has agreed
				expression = UMLFactory.eINSTANCE.createOpaqueExpression();
				constraint.setSpecification(expression);
			}
		}
		DataType datatype = UMLFactory.eINSTANCE.createDataType();
		datatype.setName(PrimitivesTypes.BOOLEAN);

		openEditor(expression, datatype);
		saveModel();
	}

	/**
	 * Check if the constraint is a NfpConstraint
	 * 
	 * @param constraint
	 * @return
	 */
	private boolean isNFPConstraint(Constraint constraint) {
		boolean isNFPConstraint = false;

		EList appliedStereotypes = constraint.getAppliedStereotypes();

		for (Object object : appliedStereotypes) {
			Stereotype stereotype = (Stereotype) object;
			if (stereotype.getName().equals("NfpConstraint")) {
				isNFPConstraint = true;
				break;
			}
		}

		return isNFPConstraint;
	}

	/**
	 * 
	 * @param slot
	 */
	public void handleSlot(Slot slot) {

		Shell shell = new Shell();

		DataType datatype = null;
		OpaqueExpression opaqueExpression = null;

		EList values = slot.getValues();

		if (values.size() == 0) {
			opaqueExpression = UMLFactory.eINSTANCE.createOpaqueExpression();
			slot.getValues().add(opaqueExpression);
		} else if (values.size() == 1) {
			if (values.get(0) instanceof OpaqueExpression) {
				opaqueExpression = (OpaqueExpression) values.get(0);
			} else {
				// we prompt user
				boolean openConfirm = MessageDialog
						.openConfirm(
								shell,
								"VSL Edition",
								"Another specification that is not VSL does exist for this slot, do you want to replace it?");
				if (!openConfirm)
					return;
				values.clear();
				opaqueExpression = UMLFactory.eINSTANCE
						.createOpaqueExpression();
				values.add(opaqueExpression);
			}
		} else if (values.size() > 1) {
			boolean openConfirm = MessageDialog
					.openConfirm(
							shell,
							"VSL Edition",
							"Another specifications that is not VSL does exist for this slot, do you want to replace it?");
			if (!openConfirm)
				return;
			values.clear();
			opaqueExpression = UMLFactory.eINSTANCE.createOpaqueExpression();
			values.add(opaqueExpression);
		}

		Type type = slot.getDefiningFeature().getType();

		if (type instanceof PrimitiveType)
			datatype = (PrimitiveType) type;
		else if (type instanceof DataType)
			datatype = (DataType) type;
		else {

			MessageDialog
					.openError(
							shell,
							"VSL Editor Error",
							"the definingFeature of this slot does not have a correct type: Only a Datatype or a PrimitiveType is allowed");
			return;
		}
		openEditor(opaqueExpression, datatype);
		saveModel();
	}

	/**
	 * Handle edition of the default value of a property
	 * 
	 * @param p
	 */
	public void handleProperty(Property p) {
		IModelFacade facade = new UMLModelFacade(p.getModel(), p);
		DataType type = facade.typeof(p);
		if (type == null) {
			MessageDialog.openError(new Shell(), "VSL Editor Error",
					"only property with a Datatype as type can be Edited");
			return;
		}
		if (p.getDefault() == null)
			p.setDefault("");

		String value = p.getDefault();

		PropertyResourceFacade vslfacade = new PropertyResourceFacade(p, type);
		VSLLabelEditorDialog pled = new VSLLabelEditorDialog(new Shell(),
				vslfacade, facade);
		pled.open();
		saveModel();
	}
}
