/**
 * <copyright>
 * Thales MARTE (Copyright (c) THALES 2007 All rights reserved) is free software; you can redistribute itand/or modify
 * it under the terms of the Eclipse Public License as published in http://www.eclipse.org/legal/epl-v10.html
 
 * Thales MARTE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Eclipse Public License for more details. 
 * </copyright>
 *
 * $Id: RSAModel.java,v 1.7 2007/10/15 09:57:11 fnizou Exp $
 */
package com.thalesgroup.nfp.rsa.model;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.uml2.uml.Class;
import org.eclipse.uml2.uml.Constraint;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.LiteralSpecification;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.ValueSpecification;

import com.cea.nfp.parsers.modelgenerator.IModelFacade;
import com.cea.nfp.parsers.modelgenerator.VSLResourceFacade;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.VSL;
import com.cea.nfp.parsers.texteditor.vsldatatypes.MarteCst.MarteLib.PrimitivesTypes;
import com.cea.nfp.popup.actions.VSLLabelEditorDialog;
import com.ibm.xtools.modeler.ui.UMLModeler;
import com.thalesgroup.nfp.rsa.facade.PropertyResourceFacade;
import com.thalesgroup.nfp.rsa.facade.RSAModelFacade;
import com.thalesgroup.nfp.rsa.facade.RSAResourceFacade;
import com.thalesgroup.nfp.rsa.facade.StereotypeAttributeFacade;
import com.thalesgroup.nfp.rsa.facade.StereotypePrimitiveTypeFacade;
import com.thalesgroup.nfp.rsa.facade.Util;

/**
 * This singleton hold the concrete method of actions. This is used to separate
 * UI Control and useCase code as much as possible.
 * 
 * @author T0081227 Francois NIZOU - july 6 2007
 * 
 */
public class RSAModel {

	private RSAModel() {

	}

	private static RSAModel self = null;

	public static RSAModel instance() {
		if (self == null)
			self = new RSAModel();
		return self;
	}

	/**
	 * Open an editor on a VSL expression
	 * 
	 * @param opaqueExpression
	 * @param datatype
	 */
	public void openEditor(OpaqueExpression opaqueExpression, DataType datatype) {
		Shell shell = new Shell();

		RSAResourceFacade vslfacade = new RSAResourceFacade(opaqueExpression,
				datatype);
		RSAModelFacade facade = new RSAModelFacade(opaqueExpression.getModel(),
				(Element) opaqueExpression.eContainer());
		VSLLabelEditorDialog pled = new VSLLabelEditorDialog(shell, vslfacade,
				facade);
		pled.open();
	}

	public void handleConstraint(final Constraint constraint) {
		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							internalHandleConstraint(constraint);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}
	}

	/**
	 * 
	 * @param constraint
	 */
	protected void internalHandleConstraint(Constraint constraint) {

		Shell shell = new Shell();

		ValueSpecification specification = constraint.getSpecification();
		OpaqueExpression expression = null;

		// get opaque expression or create it
		if (specification == null) {
			expression = UMLFactory.eINSTANCE.createOpaqueExpression();
			constraint.setSpecification(expression);
		} else {
			if (specification instanceof OpaqueExpression) {
				expression = (OpaqueExpression) specification;
			} else {
				// some specification already exist but is not opaqueExpression
				// we prompt user
				boolean openConfirm = MessageDialog
						.openConfirm(
								shell,
								"VSL Edition",
								"Another specification that is not VSL does exist for this constraint, do you want to replace it?");
				if (!openConfirm)
					return;

				// user has agreed

				expression = UMLFactory.eINSTANCE.createOpaqueExpression();
				constraint.setSpecification(expression);

				// expression = UMLFactory.eINSTANCE.createOpaqueExpression();
				// constraint.setSpecification(expression);
			}
		}
		DataType datatype = UMLFactory.eINSTANCE.createDataType();
		datatype.setName(PrimitivesTypes.BOOLEAN);

		openEditor(expression, datatype);

	}

	public void handleSlot(final Slot slot) {
		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							internalHandleSlot(slot);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}
	}

	/**
	 * 
	 * @param slot
	 */
	protected void internalHandleSlot(Slot slot) {

		Shell shell = new Shell();

		DataType datatype = null;
		OpaqueExpression opaqueExpression = null;

		EList values = slot.getValues();

		if (values.size() == 0) {
			opaqueExpression = UMLFactory.eINSTANCE.createOpaqueExpression();
			slot.getValues().add(opaqueExpression);
		} else if (values.size() == 1) {
			if (values.get(0) instanceof OpaqueExpression) {
				opaqueExpression = (OpaqueExpression) values.get(0);
			} else if (values.get(0) instanceof LiteralSpecification) {
				LiteralSpecification val = (LiteralSpecification) values.get(0);
				String stringValue = val.stringValue();
				if (stringValue == null)
					stringValue = "";
				opaqueExpression = UMLFactory.eINSTANCE
						.createOpaqueExpression();
				opaqueExpression.getBodies().add(stringValue);
				opaqueExpression.getLanguages().add(VSL.VSL_LANGUAGE);
				slot.getValues().clear();
				slot.getValues().add(opaqueExpression);
			} else {
				// we prompt user
				boolean openConfirm = MessageDialog
						.openConfirm(
								shell,
								"VSL Edition",
								"Another specification that is not VSL does exist for this slot, do you want to replace it?");
				if (!openConfirm)
					return;
				values.clear();
				opaqueExpression = UMLFactory.eINSTANCE
						.createOpaqueExpression();
				values.add(opaqueExpression);
			}
		} else if (values.size() > 1) {
			boolean openConfirm = MessageDialog
					.openConfirm(
							shell,
							"VSL Edition",
							"Another specifications that is not VSL does exist for this slot, do you want to replace it?");
			if (!openConfirm)
				return;
			values.clear();
			opaqueExpression = UMLFactory.eINSTANCE.createOpaqueExpression();
			values.add(opaqueExpression);
		}

		Type type = slot.getDefiningFeature().getType();

		if (type instanceof PrimitiveType)
			datatype = (PrimitiveType) type;
		else if (type instanceof DataType)
			datatype = (DataType) type;
		else {

			MessageDialog
					.openError(
							shell,
							"VSL Editor Error",
							"the definingFeature of this slot does not have a correct type: Only a Datatype or a PrimitiveType is allowed");
			return;
		}
		openEditor(opaqueExpression, datatype);

	}

	public void internalHandleStereotypeProperty(Element element,
			Stereotype stereotype, Property property, Integer index) {
		Object value = element.getValue(stereotype, property.getName());

		VSLResourceFacade vslfacade = null;
		RSAModelFacade facade = new RSAModelFacade(element.getModel(), element);

		if (property.getType() instanceof Class && ! property.isMultivalued()) {
			if (value == null) {
				DynamicEObjectImpl object = Util.createInstance(element,
						stereotype, property.getName());
				element.setValue(stereotype, property.getName(), object);
				value = object;
			}
			vslfacade = new StereotypeAttributeFacade((EObject) value, facade
					.typeof(property));
		} else if (property.getType() instanceof PrimitiveType) {
			vslfacade = new StereotypePrimitiveTypeFacade(element, stereotype, property);
		} else if (value instanceof EList) {
			EList eList = (EList) value;
			Object object = eList.get(index);
			if (object == null) {
				object = Util.createInstance(element, stereotype, property
						.getName());
				eList.set(index, object);
			}
			value = object;
			vslfacade = new StereotypeAttributeFacade((EObject) value, facade
					.typeof(property));
		}

		Shell shell = new Shell();
		VSLLabelEditorDialog pled = new VSLLabelEditorDialog(shell, vslfacade,
				facade);
		pled.open();
	}

	public void HandleStereotypeProperty(final Element element,
			final Stereotype stereotype, final Property property,
			final Integer index) {

		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							internalHandleStereotypeProperty(element,
									stereotype, property, index);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}
	}

	public void addArrayEntry(final Element element,
			final Stereotype stereotype, final Property property) {

		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							EList lst = (EList) element.getValue(stereotype,
									property.getName());
							DynamicEObjectImpl instance = Util.createInstance(
									element, stereotype, property.getName());
							lst.add(instance);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}

	}

	public void removeArrayEntry(final Element element,
			final Stereotype stereotype, final Property property,
			final int index) {

		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							EList lst = (EList) element.getValue(stereotype,
									property.getName());
							lst.remove(index);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}
	}

	/**
	 * 
	 * @param property
	 */
	public void handleProperty(final Property property) {
		try {
			TransactionalEditingDomain editDomain = UMLModeler
					.getEditingDomain();
			editDomain.getCommandStack().execute(
					new RecordingCommand(editDomain) {
						protected void doExecute() {
							internalHandleproperty(property);
						}
					});
		} catch (IllegalStateException e) {
			System.out.println("The operation was interrupted");
		}
	}

	protected void internalHandleproperty(Property property) {
		IModelFacade facade = new RSAModelFacade(property.getModel(), property);

		DataType type = facade.typeof(property);
		if (!(type instanceof DataType)) {
			MessageDialog
					.openConfirm(new Shell(), "VSL editor",
							"the property can not be edited for its type is not a Datatype");
			return;
		}

		String value = property.getDefault();
		value = (value == null) ? "" : value;

		VSLResourceFacade vslfacade = new PropertyResourceFacade(property, type);
		VSLLabelEditorDialog pled = new VSLLabelEditorDialog(new Shell(),
				vslfacade, facade);
		pled.open();
	}

}
